import { validateFileMetadata } from '../AI-PipeLine/config/fileValidation.js';
import sharp from 'sharp';
import fs from 'fs';
import path from 'path';
import dotenv from "dotenv";

/**
 * Validate file based on reference type and file metadata
 * @param {string} ref - Reference type
 * @param {Object} file - File object with mimetype and other properties
 * @returns {Promise<Object>} Validated file object
 * @throws {Error} If validation fails
 */
dotenv.config();

export async function validateFile(ref, file) {
    const __dirname = process.env.DIR_PATH;
    const UPLOAD_PATH = process.env.UPLOAD_PATH || '\\uploads';

    if (!file) {
        throw new Error('No file provided');
    } else {
        switch (ref) {
            case 'agents': {
                // Use the AI pipeline validation for agent files
                const validatedFile = await validateFileMetadata(file);
                if (!validatedFile) {
                    throw new Error('File validation failed for agent');
                }
                return validatedFile;
            }
            case 'users': {
                const allowedTypes = ['image/jpeg', 'image/png', 'image/jpg'];
                if (!allowedTypes.includes(file.mimetype)) {
                    throw new Error('Invalid file type. Only JPG, JPEG, and PNG are allowed.');
                }

                if (!Buffer.isBuffer(file.buffer)) {
                    throw new Error('Invalid file buffer');
                }
                const image = sharp(file.buffer);

                const metadata = await image.metadata();
                if (metadata.width > 10000 || metadata.height > 10000) {
                    throw new Error('Image resolution exceeds 10000x10000 pixels.');
                }

                // Convert to JPG if not already
                if (file.mimetype !== 'image/jpeg') {
                    image.toFormat('jpeg');
                }

                // Resize to 700x700
                const tempDir = path.join(__dirname, UPLOAD_PATH, 'temp');
                if (!fs.existsSync(tempDir)) {
                    fs.mkdirSync(tempDir, { recursive: true });
                }
                const resizedImagePath = path.join(tempDir, 'profile.jpg');
                await image.resize(700, 700).toFile(resizedImagePath);

                const resizedImageBuffer = await image.toBuffer();
                return {
                    ...file,
                    path: resizedImagePath,
                    buffer: resizedImageBuffer,
                    size: resizedImageBuffer.length,
                    mimetype: 'image/jpeg',
                };
            }
            default: {

            }
        }
    }

    // For other types, perform basic validation
    if (!file.mimetype) {
        throw new Error('File type cannot be determined');
    }

    return file;
}
