import path from 'path';

/**
 * Format successful file upload response
 * @param {Object} file - File object
 * @param {string} finalPath - Final path where file is stored
 * @param {string} fileUrl - URL to access the file
 * @param {string} ref - Reference type
 * @param {string} ref_id - Reference ID
 * @param {Object} uploadTracker - Upload tracking object
 * @returns {Object} Formatted response object
 */
export function formatSuccessResponse(file, finalPath, fileUrl, ref, ref_id, uploadTracker) {
    return {
        success: true,
        message: 'File uploaded successfully',
        data: {
            file: {
                originalName: file.originalname,
                fileName: path.basename(finalPath),
                mimeType: file.mimetype,
                size: file.size,
                url: fileUrl,
                ref,
                ref_id,
                uploadedAt: new Date().toISOString()
            },
            uploadStats: {
                processingTime: uploadTracker.processingTime,
                status: uploadTracker.status
            }
        }
    };
}

/**
 * Format error response for file upload
 * @param {Error} error - Error object
 * @param {boolean} isDevelopment - Whether in development environment
 * @returns {Object} Formatted error response
 */
export function formatErrorResponse(error, isDevelopment = false) {
    const baseResponse = {
        success: false,
        message: 'File upload failed'
    };

    // Handle required fields error
    if (error.message.includes('required')) {
        return {
            ...baseResponse,
            error: error.message
        };
    }

    // Handle invalid ref type
    if (error.message.includes('Invalid ref type')) {
        return {
            ...baseResponse,
            error: `Invalid ref type. Must be one of: organizations, users, exhibitions, boughten_booths, video_walls, posters, monitors, stands, agents`,
            validRefs: [
                'organizations',
                'users',
                'exhibitions',
                'boughten_booths',
                'video_walls',
                'posters',
                'monitors',
                'stands',
                'agents'
            ]
        };
    }

    // Handle other errors
    return {
        ...baseResponse,
        error: isDevelopment ? error.message : 'Internal server error'
    };
}