import jwt from "jsonwebtoken";
import { authConfig } from "../config/auth.js";
import { UserModel } from "../models/users.js";
import { CodeVerifyModel } from "../models/code_verifies.js";
import { sendVerificationCode } from "./sms.js";
import { generateVerificationCode } from "../utils/validation.js";

// Match original implementation more closely
export const handleSendCode = async (phone,metaboard_access) => {
  const users = await UserModel.countDocuments({ phone });
  
  // Check for existing codes in last 2 minutes, exactly as original
  const verify_codes = await CodeVerifyModel.countDocuments({
    phone,
    createdAt: { $gte: new Date(Date.now() - 2 * 60 * 1000) }
  });

  if (verify_codes > 0) {
    return {
      status: 429,
      data: null,
      message: "Too many requests. Please try again later."
    };
  }

  const code = generateVerificationCode();
  const smsSent = await sendVerificationCode(phone, code);

  if (!smsSent) {
    return {
      status: 500,
      data: null,
      message: "Server Error"
    };
  }

  // Create new user if doesn't exist
  if (users === 0) {
    await UserModel.create({
      phone:phone,
      metaboard_access:metaboard_access
    });
  }else{
    await UserModel.updateOne({
      phone:phone
    },{
      metaboard_access:metaboard_access
    })
  }

  // Save verification code
  await CodeVerifyModel.create({ phone, code });

  return {
    status: 200,
    data: null,
    message: users === 0 ? "Created Account" : "Login"
  };
};

export const handleCheckCode = async (phone, code) => {
  const user = await UserModel.findOne({ phone });
  if (!user) {
    return {
      status: 401,
      data: null,
      message: "phone_number_is_not_exist"
    };
  }

  // Match original verification logic exactly
  const verify_codes = await CodeVerifyModel.countDocuments({
    phone,
    code,
    createdAt: { $gte: new Date(Date.now() - 10 * 60 * 1000) }
  });

  if (verify_codes === 0) {
    return {
      status: 406,
      data: null,
      message: "not_acceptable_verify_code"
    };
  }

  // Generate JWT token with exact same payload as original
  const token = jwt.sign(
    {
      id: user._id,
      phone: user.phone,
      createdAt: user.createdAt
    },
    authConfig.jwtSecret,
    { expiresIn: "90d" }
  );

  // Clean up used codes
  await CodeVerifyModel.deleteMany({ phone });

  return {
    status: 200,
    data: { token },
    message: "OK"
  };
};