import express from "express";
import { body, validationResult, param } from "express-validator";
import { InvoiceModel } from "../models/invoice.js";
import { CurrenciesModel } from "../models/currencies.js";
import { PaymentMethodModel } from "../models/payment_method.js";
import { CheckCryptoPayment } from "../middleware/checkCrypto.js";
import {
  authenticateJWT,
  authenticateOrigin,
  authenticateUserAgent,
  authenticateUserExist,
  authenticateUserCompletedInformation,
  authenticateInvoice,
  authenticateUserIsAdmin,
} from "../middleware/auth.js";
import dotenv from "dotenv";
import { BoughtenBoothsModel } from "../models/boughten_booths.js";

dotenv.config();
const router = express.Router();

router.get(
  "/prepayment/:invoice_id",
  [
    param("invoice_id")
      .isMongoId()
      .withMessage("invoice_id_should_be_id")
      .notEmpty()
      .withMessage("invoice_id_cant_be_empty")
      .trim()
      .escape(),
  ],
  authenticateOrigin,
  authenticateUserAgent,
  authenticateJWT,
  authenticateUserExist,
  authenticateUserCompletedInformation,
  async (req, res) => {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return res.status(400).json({
        data: null,
        errors: errors.array(),
        message: "Validation error",
      });
    }
    const { invoice_id } = req.params;
    try {
      const invoice = await InvoiceModel.findOne({ _id: invoice_id });
      if (invoice) {
        const payment_method = await PaymentMethodModel.findOne({
          _id: invoice.payment_method_id,
        });
        if (payment_method) {
          const currency = await CurrenciesModel.findOne({
            _id: payment_method.currency_id,
          });
          if (currency) {
            if (currency.country === "CRYPTO") {
              const data = {
                invoice: invoice,
                payment_method: payment_method,
                currency: currency,
                payment_link: `https://${process.env.METABOARD_DOMAIN}/payments/crypto/${invoice_id}`,
              };
              return res.status(200).json({
                data: data,
                message: "ds",
              });
            } else {
              //Zarinpal
            }
          } else {
            return res.status(404).json({
              data: null,
              errors: [
                {
                  type: "not_found",
                  msg: "currency_not_found",
                  path: "currency",
                  location: "body",
                },
              ],
              message: null,
            });
          }
        } else {
          return res.status(404).json({
            data: null,
            errors: [
              {
                type: "not_found",
                msg: "payment_method_not_found",
                path: "payment_method",
                location: "body",
              },
            ],
            message: null,
          });
        }
      } else {
        return res.status(404).json({
          data: null,
          errors: [
            {
              type: "not_found",
              msg: "invoice_not_found",
              path: "invoice",
              location: "body",
            },
          ],
          message: null,
        });
      }
    } catch (error) {
      console.error("Error fetching :", error);
      return res
        .status(500)
        .json({ data: null, message: "Internal server error" });
    }
  }
);
router.get(
  "/test_hash",
  authenticateOrigin,
  authenticateUserAgent,
  authenticateJWT,
  authenticateUserExist,
  authenticateUserCompletedInformation,
  authenticateUserIsAdmin,
  async (req, res) => {
    res.status(200).json({
      data: {
        hash: "d5135e1dc614863342366da73b1284f159e4d1076a82a4b8d1c283bb7c4293ee",
      },
      message: "ds",
    });
  }
);
router.delete(
  "/test_hash",
  [
    body("hash")
      .isHash("sha256")
      .withMessage("hash_not_acceptable")
      .trim()
      .escape(),
  ],
  authenticateOrigin,
  authenticateUserAgent,
  authenticateJWT,
  authenticateUserExist,
  authenticateUserCompletedInformation,
  authenticateUserIsAdmin,
  async (req, res) => {
    const invoice = await InvoiceModel.findOne({ hash: req.body.hash });
    if (invoice) {
      await BoughtenBoothsModel.findOneAndDelete({
        _id: invoice.ref_id,
      });
      await InvoiceModel.findOneAndDelete({ hash: req.body.hash });
      res.sendStatus(200);
    } else {
      res.sendStatus(404);
    }
  }
);
router.get(
  "/",
  authenticateOrigin,
  authenticateUserAgent,
  authenticateJWT,
  authenticateUserExist,
  authenticateUserCompletedInformation,
  authenticateUserIsAdmin, // فقط برای ادمین‌ها – در صورت نیاز حذف کن
  async (req, res) => {
    try {
      const invoices = await InvoiceModel.find()
        .sort({ createdAt: -1 })
        .lean();

      res.status(200).json({
        data: invoices,
        message: "invoices fetched successfully",
      });
    } catch (err) {
      console.error("❌ Error fetching invoices:", err);
      res.status(500).json({
        data: null,
        message: "Internal server error",
      });
    }
  }
);

router.post(
  "/crypto",
  [
    body("invoice_id")
      .isMongoId()
      .withMessage("invoice_id_should_be_id")
      .trim()
      .escape(),
    body("hash")
      .isHash("sha256")
      .withMessage("hash_not_acceptable")
      .trim()
      .escape(),
  ],
  authenticateOrigin,
  authenticateUserAgent,
  authenticateJWT,
  authenticateUserExist,
  authenticateUserCompletedInformation,
  authenticateInvoice,
  async (req, res) => {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return res.status(400).json({
        data: null,
        ...errors,
        message: "validation error",
      });
    }
    if (req.invoice.status === "pending") {
      const payment_method = await PaymentMethodModel.findOne({
        _id: req.invoice.payment_method_id,
        is_removed: false,
      });
      if (payment_method) {
        const hash_used = await InvoiceModel.countDocuments({
          hash: req.body.hash,
        });
        if (hash_used === 0) {
          const payment_respone = await CheckCryptoPayment(
            payment_method.network,
            req.body.hash
          );
          if (payment_respone.statusCode === 200) {
            const payment_data = payment_respone.body;
            if (payment_data.hasOwnProperty("hash")) {
              const paid =
                parseFloat(payment_data.tokenTransferInfo.amount_str) / 1000000;
              const to_address = payment_data.tokenTransferInfo.to_address;
              const coin = payment_data.tokenTransferInfo.symbol;
              const network = payment_data.tokenTransferInfo.tokenType
                .toString()
                .toUpperCase();
              const currency = await CurrenciesModel.findOne({
                _id: payment_method.currency_id,
                is_removed: false,
              });
              if (currency) {
                if (
                  payment_method.address.toLowerCase() ===
                  to_address.toLowerCase()
                ) {
                  if (currency.title.toLowerCase() === coin.toLowerCase()) {
                    if (
                      payment_method.network.toLowerCase() ===
                      network.toLowerCase()
                    ) {
                      const should_to_paid =
                        req.invoice.amount -
                        req.invoice.discount +
                        req.invoice.tax;
                      if (should_to_paid === paid) {
                        const updatedInvoice =
                          await InvoiceModel.findOneAndUpdate(
                            {
                              _id: req.invoice._id,
                            },
                            {
                              status: "paid",
                              hash: req.body.hash,
                            },
                            { new: true }
                          );
                        if (updatedInvoice.ref === "boughten_booths") {
                          await BoughtenBoothsModel.updateOne(
                            { _id: updatedInvoice.ref_id },
                            { active: true }
                          );
                        }
                        res.status(200).json({
                          data: {
                            redirect: `https://${process.env.METABOARD_DOMAIN}/payments/status?success=1`,
                          },
                          message: "ds",
                        });
                      } else {
                        res.status(406).json({
                          data: null,
                          errors: [
                            {
                              type: "field",
                              msg: "amount_didnt_match",
                              path: "Not Found",
                              location: "body",
                            },
                          ],
                          message: "validation error",
                        });
                      }
                    } else {
                      res.status(406).json({
                        data: null,
                        errors: [
                          {
                            type: "field",
                            msg: "network_didnt_match",
                            path: "Not Found",
                            location: "body",
                          },
                        ],
                        message: "validation error",
                      });
                    }
                  } else {
                    res.status(406).json({
                      data: null,
                      errors: [
                        {
                          type: "field",
                          msg: "coin_didnt_match",
                          path: "Not Found",
                          location: "body",
                        },
                      ],
                      message: "validation error",
                    });
                  }
                } else {
                  res.status(406).json({
                    data: null,
                    errors: [
                      {
                        type: "field",
                        msg: "address_didnt_match",
                        path: "Not Found",
                        location: "body",
                      },
                    ],
                    message: "validation error",
                  });
                }
              } else {
                res.status(500).json({
                  data: null,
                  errors: [
                    {
                      type: "field",
                      msg: "currency_not_found",
                      path: "Not Found",
                      location: "body",
                    },
                  ],
                  message: "validation error",
                });
              }
            } else {
              res.sendStatus(500);
            }
          } else {
            res.sendStatus(500);
          }
        } else {
          res.status(406).json({
            data: null,
            errors: [
              {
                type: "field",
                msg: "hash_repetitive",
                path: "Hash Repetitive",
                location: "body",
              },
            ],
            message: "validation error",
          });
        }
      } else {
        return res.status(404).json({
          data: null,
          errors: [
            {
              type: "not_found",
              msg: "payment_method_not_found",
              path: "payment_method",
              location: "body",
            },
          ],
          message: null,
        });
      }
    } else {
      res.status(406).json({
        data: null,
        errors: [
          {
            type: "field",
            msg: "invoice_paid",
            path: "Invoice paid",
            location: "body",
          },
        ],
        message: "validation error",
      });
    }
  }
);

export default router;
