import express from "express";
import { body, query, validationResult } from "express-validator";
import mongoose from "mongoose";
import { MeetingModel } from "../models/meeting.js";
import { authenticateJWT } from "../middleware/auth.js";

const router = express.Router();

/**
 * رزرو جلسه
 */
router.post(
    "/reserve",
    [
        body("booth_id").notEmpty().withMessage("booth_id_required"),
        body("start_time").isISO8601().toDate().withMessage("start_time_invalid"),
        body("end_time").isISO8601().toDate().withMessage("end_time_invalid"),
        body("link").optional().isURL().withMessage("link_invalid"),
        body("user_id").notEmpty().withMessage("user_id_required"),
    ],
    async (req, res) => {
        const errors = validationResult(req);
        if (!errors.isEmpty()) {
            return res.status(400).json({ errors: errors.array() });
        }

        const { booth_id, user_id, start_time, end_time, link } = req.body;

        if (start_time >= end_time) {
            return res.status(400).json({ message: "end_time_must_be_after_start_time" });
        }

        try {
            // چک کردن تداخل زمان
            const conflict = await MeetingModel.findOne({
                booth_id: new mongoose.Types.ObjectId(booth_id),
                is_canceled: false,
                $or: [
                    { start_time: { $lt: end_time, $gte: start_time } },
                    { end_time: { $gt: start_time, $lte: end_time } },
                    { start_time: { $lte: start_time }, end_time: { $gte: end_time } },
                ],
            });

            if (conflict) {
                return res.status(409).json({ message: "time_conflict_with_existing_meeting" });
            }

            const meeting = new MeetingModel({
                booth_id: new mongoose.Types.ObjectId(booth_id),
                user_id: new mongoose.Types.ObjectId(user_id),
                start_time,
                end_time,
                link,
            });

            await meeting.save();

            res.status(201).json({ message: "meeting_reserved", meeting });
        } catch (error) {
            console.error(error);
            res.status(500).json({ message: "internal_server_error", error: error.message });
        }
    }
);

/**
 * کنسل کردن جلسه
 */

router.post("/cancel", authenticateJWT, async (req, res) => {
    try {
        const { meeting_id } = req.body;
        const user_id = req.user.id; // user_id از توکن JWT استخراج شده

        if (!meeting_id) {
            return res.status(400).json({ message: "meeting_id_required" });
        }

        const meeting = await MeetingModel.findOne({
            _id: new mongoose.Types.ObjectId(meeting_id),
            user_id: new mongoose.Types.ObjectId(user_id),
            is_canceled: false,
        });

        if (!meeting) {
            return res.status(404).json({ message: "meeting_not_found_or_no_permission_or_already_canceled" });
        }

        meeting.is_canceled = true;
        await meeting.save();

        return res.status(200).json({ message: "meeting_cancelled_successfully" });
    } catch (error) {
        console.error(error);
        return res.status(500).json({ message: "internal_server_error", error: error.message });
    }
});

router.post(
    "/change-time",
    [
        body("meeting_id").notEmpty().withMessage("meeting_id_required"),
        body("start_time").isISO8601().toDate().withMessage("start_time_invalid"),
        body("end_time").isISO8601().toDate().withMessage("end_time_invalid"),
    ],
    async (req, res) => {
        const errors = validationResult(req);
        if (!errors.isEmpty()) return res.status(400).json({ errors: errors.array() });

        const { meeting_id, start_time, end_time } = req.body;

        if (start_time >= end_time) {
            return res.status(400).json({ message: "end_time_must_be_after_start_time" });
        }

        try {
            const meeting = await MeetingModel.findById(meeting_id);
            if (!meeting) return res.status(404).json({ message: "meeting_not_found" });
            if (meeting.is_canceled) return res.status(400).json({ message: "meeting_already_canceled" });

            // چک تداخل با جلسات دیگه
            const conflict = await MeetingModel.findOne({
                booth_id: meeting.booth_id,
                _id: { $ne: meeting._id },
                is_canceled: false,
                $or: [
                    { start_time: { $lt: end_time, $gte: start_time } },
                    { end_time: { $gt: start_time, $lte: end_time } },
                    { start_time: { $lte: start_time }, end_time: { $gte: end_time } },
                ],
            });

            if (conflict) {
                return res.status(409).json({ message: "time_conflict_with_existing_meeting" });
            }

            meeting.start_time = start_time;
            meeting.end_time = end_time;
            await meeting.save();

            res.json({ message: "meeting_time_changed", meeting });
        } catch (error) {
            console.error(error);
            res.status(500).json({ message: "internal_server_error", error: error.message });
        }
    }
);

/**
 * لیست جلسات یک کاربر (یا برای یک booth)
 * query params: user_id یا booth_id اختیاری
 */
router.get(
    "/list",
    [
        query("user_id").optional().isString(),
        query("booth_id").optional().isString(),
    ],
    async (req, res) => {
        try {
            const filter = { is_canceled: false };

            if (req.query.user_id) {
                filter.user_id = new mongoose.Types.ObjectId(req.query.user_id);
            }
            if (req.query.booth_id) {
                filter.booth_id = new mongoose.Types.ObjectId(req.query.booth_id);
            }

            const meetings = await MeetingModel.find(filter).sort({ start_time: 1 });

            res.json({ meetings });
        } catch (error) {
            console.error(error);
            res.status(500).json({ message: "internal_server_error", error: error.message });
        }
    }
);

export default router;
