import express from "express";
import { body, validationResult, param } from "express-validator";
import { LanguageModel } from "../models/languages.js";
import { SentencesModel } from "../models/sentences.js";
import {
  authenticateJWT,
  authenticateOrigin,
  authenticateUserAgent,
  authenticateUserExist,
  authenticateUserIsOwner,
} from "../middleware/auth.js";
import { translateText } from "../services/language.js";
const router = express.Router();

/**
 * @swagger
 * components:
 *   schemas:
 *     Language:
 *       type: object
 *       required:
 *         - language
 *         - isoCode
 *       properties:
 *         language:
 *           type: string
 *           description: The name of the language
 *         isoCode:
 *           type: string
 *           description: ISO code identifier for the language
 *         sentences:
 *           type: object
 *           description: Collection of sentences in this language
 *       example:
 *         language: "English"
 *         isoCode: "EN"
 *         sentences: {}
 *     Error:
 *       type: object
 *       properties:
 *         data:
 *           type: null
 *         message:
 *           type: string
 *           description: Error message
 *
 * tags:
 *   - name: Languages
 *     description: Endpoints for managing and retrieving language information
 */
/**
 * @swagger
 * /languages:
 *   get:
 *     summary: Get all languages
 *     tags: [Languages]
 *     security:
 *       - BearerAuth: []
 *     responses:
 *       200:
 *         description: List of all languages
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 data:
 *                   type: array
 *                   items:
 *                     type: object
 *                     properties:
 *                       language:
 *                         type: string
 *                       isoCode:
 *                         type: string
 *                 message:
 *                   type: string
 *       404:
 *         description: No languages found
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Error'
 *       500:
 *         description: Internal server error
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Error'
 */
/**
 * @swagger
 * /languages/{isoCode}:
 *   get:
 *     summary: Get language by ISO code
 *     tags: [Languages]
 *     security:
 *       - BearerAuth: []
 *     parameters:
 *       - in: path
 *         name: isoCode
 *         schema:
 *           type: string
 *         required: true
 *         description: ISO code of the language to retrieve
 *     responses:
 *       200:
 *         description: Language details retrieved successfully
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 data:
 *                   $ref: '#/components/schemas/Language'
 *                 message:
 *                   type: string
 *       400:
 *         description: Invalid ISO code provided
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Error'
 *       404:
 *         description: Language not found
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Error'
 *       500:
 *         description: Internal server error
 *         content:
 *           application/json:
 *             schema:
 *               $ref: '#/components/schemas/Error'
 */

router.get("/", authenticateOrigin, authenticateUserAgent, async (req, res) => {
  try {
    const languages = await LanguageModel.find(
      {},
      { iso_code: 1, language: 1, country: 1, direction: 1, _id: 0 }
    );

    if (languages.length === 0) {
      return res.status(404).json({
        data: null,
        message: "No ISO codes found",
      });
    }

    return res.status(200).json({
      data: languages,
      message: "ds",
    });
  } catch (error) {
    console.error("Error fetching languages:", error);
    return res
      .status(500)
      .json({ data: null, message: "Internal server error" });
  }
});
router.get(
  "/key_or_value_exist/:text",
  param("text")
    .notEmpty()
    .withMessage("text is required")
    .trim()
    .escape()
    .isString()
    .withMessage("text must be a string"),
  authenticateOrigin,
  authenticateUserAgent,
  async (req, res) => {
    const errors = validationResult(req);

    if (!errors.isEmpty()) {
      return res.status(400).json({
        data: null,
        errors: errors.array(),
        message: "Validation error",
      });
    }

    let { text } = req.params;

    const foundSentences_key = await SentencesModel.find({ key: text });
    const foundSentences_value = await SentencesModel.find({ value: text });

    let languages = [];

    for (const foundSentence_key of foundSentences_key) {
      const language = await LanguageModel.findById(
        foundSentence_key.language_id
      );
      if (!languages.includes(language.iso_code)) {
        languages.push(language.iso_code);
      }
    }
    for (const foundSentence_value of foundSentences_value) {
      const language = await LanguageModel.findById(
        foundSentence_value.language_id
      );
      if (!languages.includes(language.iso_code)) {
        languages.push(language.iso_code);
      }
    }

    return res.status(200).json({
      data: {
        exist: foundSentences_key || foundSentences_value ? true : false,
        count: foundSentences_key.length + foundSentences_value.length,
        languages_lenght: languages.length,
        languages: languages,
      },
      message: "ds",
    });
  }
);
router.get(
  "/:isoCode",
  param("isoCode")
    .notEmpty()
    .withMessage("ISO code is required")
    .trim()
    .escape()
    .isString()
    .withMessage("ISO code must be a string"),
  authenticateOrigin,
  authenticateUserAgent,
  async (req, res) => {
    const errors = validationResult(req);

    if (!errors.isEmpty()) {
      return res.status(400).json({
        data: null,
        errors: errors.array(),
        message: "Validation error",
      });
    }

    let { isoCode } = req.params;

    isoCode = isoCode.toUpperCase();

    try {
      const foundLanguage = await LanguageModel.findOne(
        { iso_code: isoCode },
        { _id: 1, __v: 0, is_removed: 0, sentences: 0 }
      );

      if (!foundLanguage) {
        return res.status(404).json({
          data: null,
          message: `Language with this ISO code not found`,
        });
      }
      const sentences = await SentencesModel.find({
        language_id: foundLanguage._id,
      });
      sentences.forEach((sentence) => {
        foundLanguage.set(`sentences.${sentence.key}`, sentence.value);
      });
      return res.status(200).json({ data: foundLanguage, message: "ds" });
    } catch (error) {
      console.error("Error fetching language:", error);
      return res.status(500).json({ data: null, message: "ise" });
    }
  }
);
router.post(
  "/add_sentences",
  body("key")
    .isString()
    .withMessage("key_should_be_string")
    .notEmpty()
    .withMessage("key_cant_be_empty")
    .trim()
    .escape(),
  body("value")
    .isString()
    .withMessage("value_should_be_string")
    .notEmpty()
    .withMessage("value_cant_be_empty")
    .trim()
    .escape(),
  body("language")
    .isString()
    .withMessage("language_should_be_string")
    .notEmpty()
    .withMessage("language_cant_be_empty")
    .trim()
    .escape(),
  authenticateOrigin,
  authenticateUserAgent,
  authenticateJWT,
  authenticateUserExist,
  authenticateUserIsOwner,
  async (req, res) => {
    const errors = validationResult(req);

    if (!errors.isEmpty()) {
      return res.status(400).json({
        data: null,
        errors: errors.array(),
        message: "Validation error",
      });
    }

    const iso_code_language = req.body.language.toUpperCase();
    const language = await LanguageModel.findOne({
      iso_code: iso_code_language,
      is_removed: false,
    });
    const was_key = await SentencesModel.findOne({ key: req.body.key });
    if (language) {
      if (!was_key) {
        const all_languages = await LanguageModel.find(
          {},
          { iso_code: 1, language: 1, _id: 0 }
        );
        const input_text = {
          language: language.language,
          iso_code: language.iso_code,
          key: req.body.key,
          value: req.body.value,
        };
        var accept_language = [];
        try {
          for (let index = 0; index < all_languages.length; index++) {
            const result = await translateText(input_text, [
              all_languages[index],
            ]);
            for (const any_translate of result) {
              const language_select = await LanguageModel.findOne({
                iso_code: any_translate.iso_code.toUpperCase(),
                is_removed: false,
              });
              if (language_select) {
                accept_language.push(any_translate.iso_code.toUpperCase());
                if (
                  iso_code_language === any_translate.iso_code.toUpperCase()
                ) {
                  await SentencesModel.create({
                    language_id: language_select._id,
                    key: req.body.key,
                    value: req.body.value,
                  });
                } else {
                  await SentencesModel.create({
                    language_id: language_select._id,
                    key: req.body.key,
                    value: any_translate.value,
                  });
                }
              }
            }
          }
          res.status(200).json({
            data: input_text,
            success_iso_codes: accept_language,
            all_successes: all_languages.length === accept_language.length,
          });
        } catch (error) {
          console.log(error);
          res.status(500).json({
            data: null,
            errors: [
              {
                type: null,
                msg: "server_error",
                path: null,
                location: "server",
              },
            ],
            message: "validation error",
          });
        }
      } else {
        res.status(409).json({
          data: null,
          errors: [
            {
              type: "field",
              msg: "language_key_has_conflict",
              path: "Language",
              location: "header",
            },
          ],
          message: "validation error",
        });
      }
    } else {
      res.status(400).json({
        data: null,
        errors: [
          {
            type: "field",
            msg: "language_is_not_valid",
            path: "Language",
            location: "header",
          },
        ],
        message: "validation error",
      });
    }
  }
);
router.delete(
  "/delete_sentences",
  body("key")
    .isString()
    .withMessage("should_be_string")
    .notEmpty()
    .withMessage("phone_number_cant_be_empty")
    .trim()
    .escape(),
  authenticateOrigin,
  authenticateUserAgent,
  authenticateJWT,
  authenticateUserExist,
  authenticateUserIsOwner,
  async (req, res) => {
    const errors = validationResult(req);

    if (!errors.isEmpty()) {
      return res.status(400).json({
        data: null,
        errors: errors.array(),
        message: "Validation error",
      });
    }
    await SentencesModel.deleteMany({ key: req.body.key });
    return res.status(200).json({ data: null });
  }
);
router.post(
  "/add_language",
  // body("key")
  //   .isString()
  //   .withMessage("should_be_string")
  //   .notEmpty()
  //   .withMessage("phone_number_cant_be_empty")
  //   .trim()
  //   .escape(),
  // body("value")
  //   .isString()
  //   .withMessage("should_be_string")
  //   .notEmpty()
  //   .withMessage("otp_code_cant_be_empty")
  //   .trim()
  //   .escape(),
  // body("language")
  //   .isString()
  //   .withMessage("should_be_string")
  //   .notEmpty()
  //   .withMessage("otp_code_cant_be_empty")
  //   .trim()
  //   .escape(),
  authenticateOrigin,
  authenticateUserAgent,
  authenticateJWT,
  authenticateUserExist,
  authenticateUserIsOwner,
  async (req, res) => {
    const errors = validationResult(req);

    if (!errors.isEmpty()) {
      return res.status(400).json({
        data: null,
        errors: errors.array(),
        message: "Validation error",
      });
    }

    // Add language
  }
);

export default router;
