import express from "express";
import { InvoiceModel } from "../models/invoice.js";
import {
    authenticateOrigin,
    authenticateUserAgent,
    authenticateJWT,
    authenticateUserExist,
    authenticateUserCompletedInformation,
    authenticateUserIsAdmin,
} from "../middleware/auth.js";

const router = express.Router();

/**
 * @swagger
 * /invoices:
 *   get:
 *     tags: [Invoices]
 *     summary: Get all invoices or filter by ref
 *     description: Returns a list of invoices. Use query param `ref=all` to fetch all invoices or filter by ref name.
 *     security:
 *       - BearerAuth: []
 *     parameters:
 *       - in: query
 *         name: ref
 *         schema:
 *           type: string
 *         required: false
 *         description: Filter invoices by ref. Use "all" to get all invoices.
 *     responses:
 *       200:
 *         description: A list of invoices
 *         content:
 *           application/json:
 *             schema:
 *               type: object
 *               properties:
 *                 data:
 *                   type: array
 *                   items:
 *                     type: object
 *                 message:
 *                   type: string
 *                   example: invoices fetched successfully
 *       500:
 *         $ref: '#/components/responses/ServerError'
 */
router.get(
    "/",
    authenticateOrigin,
    authenticateUserAgent,
    authenticateJWT,
    authenticateUserExist,
    authenticateUserCompletedInformation,
    authenticateUserIsAdmin,
    async (req, res) => {
        try {
            const { ref } = req.query;

            // ایجاد فیلتر برای جستجو
            let filter = {};
            if (ref && ref !== "all") {
                filter.ref = ref;
            }

            const invoices = await InvoiceModel.find(filter)
                .sort({ createdAt: -1 })
                .lean();

            return res.status(200).json({
                data: invoices,
                message: "invoices fetched successfully",
            });
        } catch (err) {
            console.error("❌ Error fetching invoices:", err);
            return res.status(500).json({
                data: null,
                message: "Internal server error",
            });
        }
    }
);

export default router;
