import express from "express";
import { body, validationResult } from "express-validator";
import invoiceRoutes from "./routes/invoice.routes.js"; // مسیر درست فایل
app.use("/invoices", invoiceRoutes);
import { InvoiceModel } from "../models/invoice.js";
import {
    authenticateOrigin,
    authenticateUserAgent,
    authenticateJWT,
    authenticateUserExist,
    authenticateUserCompletedInformation,
    authenticateUserIsAdmin,
} from "../middleware/auth.js";

const router = express.Router();

// POST /invoices → افزودن فاکتور جدید
router.post(
    "/",
    authenticateOrigin,
    authenticateUserAgent,
    authenticateJWT,
    authenticateUserExist,
    authenticateUserCompletedInformation,
    authenticateUserIsAdmin,
    [
        body("ref").isString().notEmpty(),
        body("ref_id").isMongoId(),
        body("payment_method_id").isMongoId(),
        body("amount").isNumeric(),
        body("discount").optional().isNumeric(),
        body("tax").optional().isNumeric(),
    ],
    async (req, res) => {
        const errors = validationResult(req);
        if (!errors.isEmpty()) {
            return res.status(400).json({
                data: null,
                errors: errors.array(),
                message: "Validation error",
            });
        }

        try {
            const newInvoice = new InvoiceModel({
                ref: req.body.ref,
                ref_id: req.body.ref_id,
                payment_method_id: req.body.payment_method_id,
                amount: req.body.amount,
                discount: req.body.discount || 0,
                tax: req.body.tax || 0,
            });

            const saved = await newInvoice.save();
            res.status(201).json({
                data: saved,
                message: "Invoice created successfully",
            });
        } catch (err) {
            console.error("❌ Error saving invoice:", err);
            res.status(500).json({
                data: null,
                message: "Internal server error",
            });
        }
    }
);

export default router;
