import express from "express";
import { query, validationResult } from "express-validator";
import fs from "fs/promises";
import path from "path";
import mongoose from "mongoose";

import { BoughtenBoothsModel } from "../models/boughten_booths.js";

import {
  authenticateOrigin,
  authenticateUserAgent,
  authenticateJWT,
  authenticateUserExist,
  authenticateUserCompletedInformation,
} from "../middleware/auth.js";

const router = express.Router();

// دانلود فایل بر اساس booth_id و category
router.get(
  "/download/",
  [
    query("booth_id").isString().withMessage("booth_id_must_be_string"),
    query("category").isString().withMessage("category_must_be_string"),
    query("direct_download")
      .optional()
      .isString()
      .withMessage("direct_download_must_be_string"),
  ],
  authenticateOrigin,
  authenticateUserAgent,
  authenticateJWT,
  authenticateUserExist,
  authenticateUserCompletedInformation,
  async (req, res) => {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return res.status(400).json({
        data: null,
        errors: errors.array(),
        message: "validation error",
      });
    }

    let booth_id = req.query.booth_id.trim();
    let category = req.query.category.trim();
    let direct_download = req.query.direct_download
      ? req.query.direct_download.trim()
      : "false";

    try {
      // بررسی اینکه booth_id معتبر هست یا نه
      let boothData;
      try {
        const boothObjectId = new mongoose.Types.ObjectId(booth_id);
        boothData = await BoughtenBoothsModel.findOne({
          _id: boothObjectId,
          is_removed: false,
        });
      } catch (e) {
        return res.status(400).json({
          data: null,
          message: "invalid_booth_id_format",
        });
      }

      if (!boothData) {
        return res.status(404).json({
          data: null,
          message: "booth_not_found",
        });
      }

      // مسیر فایل روی سرور
      const basePath = path.join(process.cwd(), "uploads");
      const boothPath = path.join(
        basePath,
        "exhibitions",
        boothData.exhibition_id.toString(),
        "boughten_booths",
        boothData._id.toString()
      );

      const validCategories = ["posters", "video_walls", "monitors", "stands", "logo"];

      if (!validCategories.includes(category)) {
        return res.status(400).json({
          data: null,
          message: "invalid_category",
        });
      }

      // گرفتن لیست فایل‌ها
      let filesInCategory = [];
      try {
        filesInCategory = await fs.readdir(path.join(boothPath, category));
      } catch (e) {
        filesInCategory = [];
      }

      if (filesInCategory.length === 0) {
        return res.status(404).json({
          data: null,
          message: "file_not_found_in_category",
        });
      }

      const fileName = filesInCategory[0];
      const fileFullPath = path.join(boothPath, category, fileName);

      if (direct_download === "true") {
        return res.download(fileFullPath, fileName);
      } else {
        const protocol = req.protocol;
        const host = req.get("host");

        const fileUrl = `${protocol}://${host}/uploads/exhibitions/${boothData.exhibition_id}/boughten_booths/${boothData._id}/${category}/${fileName}`;

        return res.status(200).json({
          data: fileUrl,
          message: "success",
        });
      }
    } catch (error) {
      console.error("Download error:", error);
      return res.status(500).json({
        data: error.message,
        message: "internal_server_error",
      });
    }
  }
);

export default router;
