import jwt from "jsonwebtoken";
import { authConfig } from "../config/auth.js";
import { UserModel } from "../models/users.js";
import { LanguageModel } from "../models/languages.js";
import { BoughtenBoothsModel } from "../models/boughten_booths.js";
import { CurrenciesModel } from "../models/currencies.js";
import { InvoiceModel } from "../models/invoice.js";
import { ConfigModel } from "../models/configs.js";
import { validationResult } from "express-validator";

export const authenticateErrors = (req, res, next) => {
  const errors = validationResult(req);
  if (!errors.isEmpty()) {
    return res.status(400).json({
      data: null,
      errors: errors.array(),
      message: "Validation error",
    });
  }
  next();
};

export const authenticateJWT = (req, res, next) => {
  const authHeader = req.get("Authorization");

  if (authHeader) {
    const token = authHeader.split(" ")[1];
    jwt.verify(token, authConfig.jwtSecret, (err, user) => {
      if (err) {
        return res.status(403).json({
          data: null,
          errors: [
            {
              type: "field",
              msg: "access_denied",
              path: "Authorization",
              location: "header",
            },
          ],
          message: "validation error",
        });
      }
      req.user = user;
      next();
    });
  } else {
    return res.status(401).json({
      data: null,
      errors: [
        {
          type: "field",
          msg: "unauthorized",
          path: "Authorization",
          location: "header",
        },
      ],
      message: "validation error",
    });
  }
};

export function authenticateOrigin(req, res, next) {
  // به همه اجازه بده
  return next();
}



export const authenticateUserAgent = (req, res, next) => {
  const userAgent = req.get("User-Agent");
  if (userAgent && userAgent.includes("Mozilla")) {
    next();
  } else {
    return res.status(403).json({
      data: null,
      errors: [
        {
          type: "field",
          msg: "access_denied",
          path: "User-Agent",
          location: "header",
        },
      ],
      message: "validation error",
    });
  }
};

export const authenticateUserExist = async (req, res, next) => {
  const user = await UserModel.findOne({ _id: req.user?.id });
  if (user) {
    req.user = user;
    next();
  } else {
    return res.status(401).json({
      data: null,
      errors: [
        {
          type: "field",
          msg: "unauthorized",
          path: "Authorization",
          location: "header",
        },
      ],
      message: "validation error",
    });
  }
};

export const authenticateUserIsAdmin = async (req, res, next) => {
  if (req.user.admin_access) {
    next();
  } else {
    return res.status(403).json({
      data: null,
      errors: [
        {
          type: "role",
          msg: "access_denied",
          path: "Authorization",
          location: "header",
        },
      ],
      message: "validation error",
    });
  }
};

export const authenticateUserIsOwner = async (req, res, next) => {
  const config_data = await ConfigModel.findOne();
  if (config_data && config_data.owners.includes(req.user._id)) {
    next();
  } else {
    return res.status(403).json({
      data: null,
      errors: [
        {
          type: "role",
          msg: "access_denied",
          path: "Authorization",
          location: "header",
        },
      ],
      message: "validation error",
    });
  }
};

export const authenticateLanguageIsoCode = async (req, res, next) => {
  const LanguageIsoCode = (req.get("Language") || "FA").toUpperCase();
  const languages = await LanguageModel.find(
    { is_removed: false },
    { iso_code: 1, language: 1, direction: 1, _id: 0 }
  );
  const matchedLanguage = languages.find(
    (lang) => lang.iso_code === LanguageIsoCode
  );
  if (matchedLanguage) {
    req.language = matchedLanguage;
    next();
  } else {
    return res.status(400).json({
      data: null,
      errors: [
        {
          type: "field",
          msg: "language_is_not_valid",
          path: "Language",
          location: "header",
        },
      ],
      message: "validation error",
    });
  }
};

export const authenticateUserCompletedInformation = async (req, res, next) => {
  if (req.user.name?.trim() && req.user.national_code?.trim()) {
    next();
  } else {
    return res.status(422).json({
      data: null,
      errors: [
        {
          type: "field",
          msg: "incomplete_information",
          path: "User",
          location: "body",
        },
      ],
      message: "validation error",
    });
  }
};

export const authenticateCurrency = async (req, res, next) => {
  const Currency = req.get("Currency");
  const CurrnecyFound = await CurrenciesModel.findOne({
    symbol: Currency,
    is_removed: false,
  });
  if (CurrnecyFound && CurrnecyFound.symbol === String(Currency)) {
    req.currency = CurrnecyFound;
  } else {
    const fallbackCurrency = await CurrenciesModel.findOne({ symbol: "crypto_usdt" });
    req.currency = fallbackCurrency;
  }
  next();
};

export const authenticateInvoice = async (req, res, next) => {
  const invoice_id = req.body.invoice_id;
  if (invoice_id) {
    const InvoiceFound = await InvoiceModel.findOne({ _id: invoice_id, is_removed: false });
    if (InvoiceFound) {
      if (InvoiceFound.ref === "boughten_booths") {
        const ref_table = await BoughtenBoothsModel.findOne({ _id: InvoiceFound.ref_id });
        if (ref_table) {
          if (ref_table.owner === "users" && ref_table.owner_id == req.user.id) {
            req.invoice = InvoiceFound;
            next();
          } else {
            return res.status(404).json({
              data: null,
              errors: [
                { type: "field", msg: "invoice_not_found", path: "Not Found", location: "body" }
              ],
              message: "validation error",
            });
          }
        } else {
          return res.status(404).json({ data: null, errors: [{ type: "field", msg: "ref_not_found", path: "Not Found", location: "body" }], message: "validation error" });
        }
      } else {
        return res.status(406).json({ data: null, errors: [{ type: "field", msg: "ref_not_accptable", path: "Not Found", location: "body" }], message: "validation error" });
      }
    } else {
      return res.status(404).json({ data: null, errors: [{ type: "field", msg: "invoice_not_found", path: "Not Found", location: "body" }], message: "validation error" });
    }
  } else {
    return res.status(404).json({ data: null, errors: [{ type: "field", msg: "invoice_id_not_found", path: "Not Found", location: "body" }], message: "validation error" });
  }
};

export const authenticateOwnerBooth = async (req, res, next) => {
  const { boughten_booth_id } = req.params;
  if (boughten_booth_id) {
    const boughten_booth_data = await BoughtenBoothsModel.findOne({
      _id: boughten_booth_id,
      owner: "users",
      owner_id: req.user.id,
      is_removed: false,
    });
    if (boughten_booth_data) {
      req.boughten_booth = boughten_booth_data;
      next();
    } else {
      return res.sendStatus(403);
    }
  } else {
    return res.status(400).json({
      data: null,
      errors: [
        {
          type: "field",
          value: ":boughten_booth_id",
          msg: "boughten_booth_id_should_be_id",
          path: "boughten_booth_id",
          location: "params",
        },
      ],
      message: "Validation error",
    });
  }
};
