import chardet from 'chardet';
import { detect } from 'langdetect';
import fileConfig from '../config/fileValidation.js';

/**
 * Utility class for analyzing file content
 */
class FileAnalysis {
    /**
     * Detects character encoding of file content
     * @param {Buffer} buffer - File content buffer
     * @returns {string} Detected encoding
     */
    static async detectEncoding(buffer) {
        try {
            // Detect encoding using chardet
            const detected = chardet.detect(buffer);
            
            // Check if detected encoding is supported
            if (fileConfig.encoding.supported.includes(detected)) {
                return detected;
            }
            
            // Return default encoding if detected encoding is not supported
            return fileConfig.encoding.default;
        } catch (error) {
            console.warn('Error detecting encoding:', error);
            return fileConfig.encoding.fallback;
        }
    }

    /**
     * Detects languages in the text content
     * @param {Buffer} buffer - File content buffer
     * @returns {Object} Object containing detected languages and RTL status
     */
    static async detectLanguage(buffer) {
        try {
            // Convert buffer to string using detected encoding
            const encoding = await FileAnalysis.detectEncoding(buffer);
            const text = buffer.toString(encoding);

            // Detect languages using langdetect
            const languages = detect(text);
            
            if (!languages || languages.length === 0) {
                return {
                    languages: [],
                    hasRtl: false
                };
            }

            // Check for RTL content (Arabic, Persian, Hebrew)
            const rtlLanguages = ['ar', 'fa', 'he'];
            const hasRtl = languages.some(lang => 
                rtlLanguages.includes(lang.lang)
            );

            // Check specifically for Persian content
            const hasPersian = languages.some(lang => lang.lang === 'fa');
            
            if (hasPersian && fileConfig.persianSupport.enabled) {
                // Normalize Persian text if needed
                const normalizedText = FileAnalysis.normalizePersianText(text);
                // Re-detect languages with normalized text to ensure accuracy
                const normalizedLangs = detect(normalizedText);
                return {
                    languages: normalizedLangs?.map(l => l.lang) || [],
                    hasRtl,
                    normalizedText
                };
            }

            return {
                languages: languages.map(l => l.lang),
                hasRtl
            };
        } catch (error) {
            console.warn('Error detecting language:', error);
            return {
                languages: [],
                hasRtl: false
            };
        }
    }

    /**
     * Normalizes Persian text according to configuration
     * @param {string} text - Text to normalize
     * @returns {string} Normalized text
     */
    static normalizePersianText(text) {
        if (!fileConfig.persianSupport.enabled) {
            return text;
        }

        let normalized = text;
        // Apply character normalizations
        Object.entries(fileConfig.persianSupport.normalizeChars).forEach(([from, to]) => {
            normalized = normalized.replace(new RegExp(from, 'g'), to);
        });

        return normalized;
    }

    /**
     * Validates text content against configured limits
     * @param {string} text - Text content to validate
     * @param {string} fileType - Type of file being validated
     * @returns {Object} Validation results
     */
    static validateContent(text, fileType) {
        const errors = [];

        try {
            // Check line length for text files
            if (['text/plain', 'text/markdown'].includes(fileType)) {
                const lines = text.split('\n');
                const longLines = lines.filter(line => 
                    line.length > fileConfig.contentValidation.maxLineLength
                );
                
                if (longLines.length > 0) {
                    errors.push(`${longLines.length} lines exceed maximum length of ${fileConfig.contentValidation.maxLineLength} characters`);
                }
            }

            // Validate JSON structure
            if (fileType === 'application/json') {
                try {
                    const json = JSON.parse(text);
                    const nestingDepth = FileAnalysis.calculateNestingDepth(json);
                    
                    if (nestingDepth > fileConfig.contentValidation.maxJsonNesting) {
                        errors.push(`JSON nesting depth ${nestingDepth} exceeds maximum of ${fileConfig.contentValidation.maxJsonNesting}`);
                    }

                    const longKeys = FileAnalysis.findLongKeys(json);
                    if (longKeys.length > 0) {
                        errors.push(`${longKeys.length} JSON keys exceed maximum length of ${fileConfig.contentValidation.maxKeyLength} characters`);
                    }
                } catch (error) {
                    errors.push(`Invalid JSON structure: ${error.message}`);
                }
            }

            return {
                isValid: errors.length === 0,
                errors
            };
        } catch (error) {
            console.error('Content validation error:', error);
            return {
                isValid: false,
                errors: ['Content validation failed: ' + error.message]
            };
        }
    }

    /**
     * Calculates the maximum nesting depth of an object
     * @param {Object} obj - Object to analyze
     * @returns {number} Maximum nesting depth
     */
    static calculateNestingDepth(obj, depth = 0) {
        if (typeof obj !== 'object' || obj === null) {
            return depth;
        }

        return Math.max(
            ...Object.values(obj).map(value => 
                FileAnalysis.calculateNestingDepth(value, depth + 1)
            )
        );
    }

    /**
     * Finds keys that exceed the maximum length
     * @param {Object} obj - Object to analyze
     * @returns {Array} Array of long keys
     */
    static findLongKeys(obj) {
        const longKeys = [];
        
        function checkKeys(o) {
            if (!o || typeof o !== 'object') return;
            
            Object.keys(o).forEach(key => {
                if (key.length > fileConfig.contentValidation.maxKeyLength) {
                    longKeys.push(key);
                }
                checkKeys(o[key]);
            });
        }

        checkKeys(obj);
        return longKeys;
    }
}

export default FileAnalysis;