import { get_encoding, encoding_for_model } from 'tiktoken';

export async function waitForCompletion(openai, threadId, runId) {
  let run = await openai.beta.threads.runs.retrieve(threadId, runId);

  while (run.status === "queued" || run.status === "in_progress") {
    await new Promise(resolve => setTimeout(resolve, 1000));
    run = await openai.beta.threads.runs.retrieve(threadId, runId);
  }

  if (run.status === "failed") {
    throw new Error("Assistant run failed: " + run.last_error);
  }

  return run;
}

export function calculateTokens(text, model = 'gpt-3.5-turbo') {
  try {
    // Get encoding for the specified model
    const enc = encoding_for_model(model);
    
    // Handle null or undefined input
    if (!text) {
      return 0;
    }

    // Ensure input is a string
    const str = String(text);
    
    // Encode the text and get token count
    const tokens = enc.encode(str);
    const count = tokens.length;
    
    // Free the encoder to prevent memory leaks
    enc.free();
    
    return count;
  } catch (error) {
    // Fallback to cl100k_base encoding if model-specific encoding fails
    try {
      const enc = get_encoding('cl100k_base');
      const tokens = enc.encode(String(text || ''));
      const count = tokens.length;
      enc.free();
      return count;
    } catch (fallbackError) {
      console.error('Token calculation failed:', fallbackError);
      // Ultimate fallback - return 0 to prevent crashing
      return 0;
    }
  }
}