import path from 'path';
import request from 'supertest';
import app from '../../app.js';

describe('Speech Integration Tests', () => {
  describe('POST /agent/speech/transcribe', () => {
    it('should transcribe audio file successfully', async () => {
      const testAudioPath = path.join(__dirname, '../test-files/test-audio.mp3');
      
      const response = await request(app)
        .post('/agent/speech/transcribe')
        .attach('file', testAudioPath)
        .expect('Content-Type', /json/)
        .expect(200);

      expect(response.body.success).toBe(true);
      expect(response.body.data.text).toBeDefined();
    });

    it('should reject invalid file types', async () => {
      const testFilePath = path.join(__dirname, '../test-files/test.txt');
      
      const response = await request(app)
        .post('/agent/speech/transcribe')
        .attach('file', testFilePath)
        .expect('Content-Type', /json/)
        .expect(400);

      expect(response.body.success).toBe(false);
      expect(response.body.error).toMatch(/Invalid file type/);
    });

    it('should handle missing file', async () => {
      const response = await request(app)
        .post('/agent/speech/transcribe')
        .expect('Content-Type', /json/)
        .expect(400);

      expect(response.body.success).toBe(false);
      expect(response.body.error).toMatch(/No audio file provided/);
    });
  });

  describe('POST /agent/speech/synthesize', () => {
    it('should synthesize speech successfully', async () => {
      const testText = 'Hello, this is a test.';
      
      const response = await request(app)
        .post('/agent/speech/synthesize')
        .send({
          text: testText,
          voice: 'alloy'
        })
        .expect('Content-Type', /audio\/mpeg/)
        .expect(200);

      expect(response.body).toBeInstanceOf(Buffer);
    });

    it('should handle missing text parameter', async () => {
      const response = await request(app)
        .post('/agent/speech/synthesize')
        .send({
          voice: 'alloy'
        })
        .expect('Content-Type', /json/)
        .expect(400);

      expect(response.body.success).toBe(false);
      expect(response.body.error).toMatch(/text is required/);
    });

    it('should validate voice parameter', async () => {
      const response = await request(app)
        .post('/agent/speech/synthesize')
        .send({
          text: 'Test',
          voice: 'invalid-voice'
        })
        .expect('Content-Type', /json/)
        .expect(400);

      expect(response.body.success).toBe(false);
      expect(response.body.error).toMatch(/Invalid voice option/);
    });
  });
});