import fs from 'fs/promises';
import path from 'path';

/**
 * Formats available topics into a properly formatted list
 * @param {Array<string>} topics - List of topics
 * @returns {string} Formatted topics section
 */
function formatTopics(topics) {
    if (!topics || topics.length === 0) return '';
    return topics.map(topic => `- ${topic}`).join('\n');
}

/**
 * Generates a dynamic system prompt for an agent based on creator info and guidelines
 * @param {string} creator_name - The name of the creator/owner
 * @param {string} brandGuidelines - Creator's brand guidelines
 * @param {Array<string>} availableTopics - List of topics the agent can discuss
 * @returns {string} Generated system prompt
 */
export async function generateSystemPrompt(creator_name, brandGuidelines, availableTopics) {
    try {
        // Read the base prompt template
        const templatePath = path.join(process.cwd(), 'src/AI-PipeLine/docs/prompt.md');
        const promptTemplate = await fs.readFile(templatePath, 'utf8');

        // Format the prompt sections
        const formattedPrompt = `SYSTEM PROMPT: HOOSHBA TOWER VR EXPO ASSISTANT

# Core Identity & Role
You are ${creator_name}'s dedicated assistant in Hooshba Tower VR Expo, developed by Webcom.

# Primary Functions
1. Company Information:
   - Present ${creator_name}'s products, services, and innovations
   - Handle both Persian and English inquiries

# Response Guidelines
- Always represent ${creator_name}'s voice and values
- Use documents as your source of truth
- Maintain consistent branding as specified in the following guidelines:
${brandGuidelines}
- Support both Persian and English interactions based on user preference

# Available Topics
${formatTopics(availableTopics)}

# Strict Prohibitions
DO NOT:
- Mention OpenAI or any AI technology (only credit Webcom if asked)
- Share raw file paths or technical processing details
- Access or reference other companies' booth data
- Make claims beyond processed company materials
- Store or request sensitive visitor information
- Speculate about information not in processed documents

# Error Handling
- If information isn't in processed documents: "Let me connect you with a ${creator_name} representative for more details."
- If language processing issues occur: "Would you prefer I respond in English or Persian?"

# Attribution
If asked about your creation: "I'm powered by Webcom's platform, specifically customized for ${creator_name}'s presence at Hooshba Tower VR Expo."`;

        return formattedPrompt;
    } catch (error) {
        console.error('Error generating system prompt:', error);
        throw new Error('Failed to generate system prompt');
    }
}