/**
 * Creates an OpenAI assistant
 * @param {Object} config Assistant configuration
 * @param {OpenAI} client OpenAI client
 * @param {function} withRetry Retry function
 * @param {number} maxRetries Maximum retry attempts
 * @param {number} retryDelay Delay between retries in milliseconds
 */
export async function createAssistant(config, client, withRetry, maxRetries, retryDelay) {
    return await withRetry(async () => {
        const assistant = await client.beta.assistants.create({
            name: config.name,
            instructions: config.instructions,
            tools: config.tools || [],
            model: config.model
        });
        return assistant;
    }, maxRetries, retryDelay);
}

/**
 * Attaches a vector store to an assistant
 */
export async function attachVectorStoreToAssistant(assistant_id, vector_store_id, client, withRetry, maxRetries, retryDelay) {
    return await withRetry(async () => {
        const assistant = await client.beta.assistants.update(
            assistant_id,
            {
                tool_resources: {
                    file_search: {
                        vector_store_ids: [vector_store_id]
                    }
                }
            }
        );
        return assistant;
    }, maxRetries, retryDelay);
}

/**
 * Creates an assistant with file search capability and attaches processed files
 */
export async function createAssistantWithFileSearch(config, client, withRetry, maxRetries, retryDelay, createVectorStore, addFilesToVectorStore, attachVectorStoreToAssistant) {
    try {
        // Create assistant with file search tool
        const assistantConfig = { ...config, tools: [{ type: 'file_search' }]
 };
        delete assistantConfig.file_ids; // Ensure file_ids isn't passed to create
        const assistant = await createAssistant(assistantConfig, client, withRetry, maxRetries, retryDelay);

        // Create vector store
        const vector_store = await createVectorStore({
            name: config.vector_store_name
        }, client, withRetry, maxRetries, retryDelay);

        // Add files to vector store if provided
        if (config.file_ids?.length > 0) {
            await addFilesToVectorStore(vector_store.id, config.file_ids, client, withRetry, maxRetries, retryDelay);
        }

        // Attach vector store to assistant
        const updated_assistant = await attachVectorStoreToAssistant(
            assistant.id,
            vector_store.id,
            client, withRetry, maxRetries, retryDelay
        );

        return {
            assistant: updated_assistant,
            vector_store
        };
    } catch (error) {
        throw new Error(`Failed to create assistant with file search: ${error.message}`);
    }
}