import OpenAI from 'openai';
import { uploadFile } from './openai/file-handler.js';
import { createAssistant, createAssistantWithFileSearch, attachVectorStoreToAssistant } from './openai/assistant-handler.js';
import { createVectorStore, addFilesToVectorStore } from './openai/vector-store-handler.js';
import { cleanup } from './openai/cleanup-handler.js';
import { withRetry } from './openai/retry-handler.js';

/**
 * Service for handling OpenAI operations in the AI Pipeline
 */
export class OpenAIService {
    /**
     * @param {string} apiKey OpenAI API key
     */
    constructor(apiKey) {
        this.client = new OpenAI({ apiKey });
        this.active_vector_stores = new Map();
        this.MAX_RETRIES = 3;
        this.RETRY_DELAY = 1000;
        this.MAX_FILE_SIZE = 25 * 1024 * 1024; // 25MB
    }

    /**
     * Upload a file to OpenAI
     * @param {string} file_path Path to the file
     * @returns {Promise<Object>} Uploaded file information
     */
    async upload_file(file_path) {
        return await uploadFile(file_path, this.client, this.MAX_FILE_SIZE, withRetry, this.MAX_RETRIES, this.RETRY_DELAY);
    }

    /**
     * Creates an OpenAI assistant
     * @param {Object} config Assistant configuration
     */
    async create_assistant(config) {
        return await createAssistant(config, this.client, withRetry, this.MAX_RETRIES, this.RETRY_DELAY);
    }

    /**
     * Creates a vector store for file search
     */
    async create_vector_store(config) {
        return await createVectorStore(config, this.client, withRetry, this.MAX_RETRIES, this.RETRY_DELAY);
    }

    /**
     * Adds processed files to a vector store
     */
    async add_files_to_vector_store(vector_store_id, file_ids) {
        return await addFilesToVectorStore(vector_store_id, file_ids, this.client, withRetry, this.MAX_RETRIES, this.RETRY_DELAY);
    }

    /**
     * Attaches a vector store to an assistant
     */
    async attach_vector_store_to_assistant(assistant_id, vector_store_id) {
        return await attachVectorStoreToAssistant(assistant_id, vector_store_id, this.client, withRetry, this.MAX_RETRIES, this.RETRY_DELAY);
    }

    /**
     * Creates an assistant with file search capability and attaches processed files
     */
    async create_assistant_with_file_search(config) {
        return await createAssistantWithFileSearch(config, this.client, withRetry, this.MAX_RETRIES, this.RETRY_DELAY, this.create_vector_store.bind(this), this.add_files_to_vector_store.bind(this), this.attach_vector_store_to_assistant.bind(this));
    }

    /**
     * Updates an existing assistant with new files and vector store
     * @param {Object} config Configuration object containing assistant_id and file_ids
     * @returns {Promise<{assistant: Object, vector_store: Object}>} Updated assistant and vector store
     */
    async add_files_to_assistant(config) {
        // Create new vector store for the files
        const vector_store = await this.create_vector_store({
            name: `documents_${config.assistant_id}`,
        });

        // Add files to vector store and attach to assistant
        await this.add_files_to_vector_store(vector_store.id, config.file_ids);
        await this.attach_vector_store_to_assistant(config.assistant_id, vector_store.id);

        return { assistant: { id: config.assistant_id }, vector_store };
    }

    /**
     * Cleanup resources associated with this service
     */
    async cleanup() {
        await cleanup(this.active_vector_stores, this.client);
    }
}