import { AgentModel } from '../models/agent.js';
import { generateSystemPrompt } from './prompt-service.js';

/**
 * Creates a new agent with dynamically generated system prompt
 * @param {Object} agentData - Agent creation data
 * @returns {Promise<Object>} Created agent document
 */
export async function createAgent(agentData) {
    try {
        const systemPrompt = await generateSystemPrompt(
            agentData.name,
            agentData.brand_guidelines,
            agentData.available_topics
        );

        const agent = new AgentModel({
            ...agentData,
            system_prompt: systemPrompt,
        });

        return await agent.save();
    } catch (error) {
        console.error('Error creating agent:', error);
        throw error;
    }
}

/**
 * Updates an agent's system prompt
 * @param {string} agent_id - The ID of the agent to update
 * @param {Object} updateData - New data to update prompt
 * @returns {Promise<Object>} Updated agent document
 */
export async function updateAgentPrompt(agent_id, updateData) {
    try {
        const agent = await AgentModel.findById(agent_id);
        if (!agent) {
            throw new Error('Agent not found');
        }

        const systemPrompt = await generateSystemPrompt(
            updateData.name || agent.name,
            updateData.brand_guidelines || agent.brand_guidelines,
            updateData.available_topics || agent.available_topics
        );

        agent.system_prompt = systemPrompt;
        if (updateData.brand_guidelines) agent.brand_guidelines = updateData.brand_guidelines;
        if (updateData.available_topics) agent.available_topics = updateData.available_topics;
        if (updateData.name) agent.name = updateData.name;

        return await agent.save();
    } catch (error) {
        console.error('Error updating agent prompt:', error);
        throw error;
    }
}