import { AgentModel } from '../models/agent.js';
import { OpenAIService } from './openai-service.js';
import { QueueManager } from '../controllers/queue/QueueManager.js';
import fs from 'fs';
import { promises as fsPromises } from 'fs';
import path from 'path';
import mongoose from 'mongoose';
import OpenAI from 'openai';

/**
 * Service for handling agent file operations
 */
export class AgentFileService {
    constructor(openai_api_key) {
        this.openai_service = new OpenAIService(openai_api_key);
        this.queueManager = new QueueManager();
        this.openai = new OpenAI({
            apiKey: openai_api_key
        });
        // Check for stalled jobs every 5 minutes
        setInterval(() => this.queueManager.handleStalledJobs(), 5 * 60 * 1000);
    }

    /**
     * Processes uploaded files for an agent
     * @param {string} agent_id - The ID of the agent
     * @param {object} upload_response - The response from the file upload controller
     * @returns {Promise<Object>} Updated agent document
     */
    async handle_agent_file_upload(agent_id, upload_response) {
        if (!upload_response.success || !upload_response.data || !upload_response.data.file) {
            throw new Error('Invalid upload response');
        }

        const file_data = upload_response.data.file;
        const agent = await AgentModel.findById(agent_id);

        if (!agent) {
            throw new Error('Agent not found');
        }

        const session = await mongoose.startSession();
        try {
            await session.withTransaction(async () => {
                // Add file to agent's raw_files with metadata
                const fileEntry = {
                    file_id: new mongoose.Types.ObjectId(),
                    file_url: file_data.url,
                    processing_status: 'pending',
                    file_metadata: {
                        original_name: file_data.original_name,
                        size: file_data.size,
                        mime_type: file_data.mime_type
                    }
                };
                agent.raw_files.push(fileEntry);

                // Verify boughten booth relationship
                if (agent.ref === 'boughten_booths') {
                    const BoughtenBoothModel = mongoose.model('boughten_booths');
                    const boughtenBooth = await BoughtenBoothModel.findById(agent.ref_id).session(session);

                    if (!boughtenBooth) {
                        throw new Error(`Boughten booth not found with ID: ${agent.ref_id}`);
                    }

                    // Create or update queue entry if needed
                    const existingQueue = await this.queueManager.getAgentQueue(agent._id);
                    if (!existingQueue && agent.queue_status === 'queued') {
                        const estimatedTime = new Date(Date.now() + (agent.raw_files.length * 60000)); // Estimate 1 min per file
                        await this.queueManager.createQueueEntry(
                            agent._id,
                            boughtenbooth.exhibition_id,
                            boughtenBooth._id,
                            estimatedTime
                        );
                    }
                }

                await agent.save({ session });
            });

            return agent;
        } catch (error) {
            console.error('Error updating agent with new file:', error);
            throw error;
        } finally {
            session.endSession();
        }
    }

    /**
     * Previously this was part of handle_agent_file_upload
     * Now separated to be triggered explicitly via build endpoint
     */
    async process_uploaded_file(agent_id, file_url) {
        try {
            const agent = await AgentModel.findById(agent_id);
            if (!agent) {
                throw new Error('Agent not found');
            }

            await this.process_and_upload_file(agent_id, file_url);
            return agent;
        } catch (error) {
            console.error('Error updating agent with new file:', error);
            throw error;
        }
    }

    /**
     * Process raw files and upload to OpenAI
     * @param {string} agent_id - The ID of the agent
     * @param {string} file_url - The URL of the file to process
     */
    async process_and_upload_file(agent_id, file_url) {
        const agent = await AgentModel.findById(agent_id);
        let uploaded_file = null;
        if (!agent) {
            throw new Error('Agent not found');
        }

        const file_entry = agent.raw_files.find(f => f.file_url === file_url);
        if (!file_entry) {
            throw new Error('File not found in agent records');
        }

        try {
            const session = await mongoose.startSession();
            // Update status to processing
            await agent.update_file_status(file_url, 'processing');

            // Get absolute file path from URL
            const file_path = path.join(process.env.UPLOAD_PATH, file_url);

            // Upload to OpenAI
            uploaded_file = await this.openai_service.upload_file(file_path);

            await session.withTransaction(async () => {
                const queueStatus = await this.queueManager.getAgentQueue(agent._id);
                if (queueStatus) {
                    queueStatus.processing_details.processed_files += 1;
                    const progress = (queueStatus.processing_details.processed_files / agent.raw_files.length) * 100;
                    await queueStatus.update_progress(progress);
                }

                // Update agent file status
                await agent.update_file_status(file_url, 'processed', {
                    openai_file_id: uploaded_file.id
                }, { session });
            });

            session.endSession();

            return {
                file_id: uploaded_file.id,
                file_url: file_url
            };
        } catch (error) {
            console.error('Error processing file:', error);

            // Cleanup any partially uploaded file
            if (uploaded_file && uploaded_file.id) {
                try {
                    await this.openai_service.client.files.delete(uploaded_file.id);
                } catch (cleanupError) {
                    console.error('Error cleaning up failed upload:', cleanupError);
                }
            }

            // Log the processing attempt and error
            await agent.log_processing_attempt({
                error_message: error.message,
                error_type: error.name || 'ProcessingError',
                file_url,
                timestamp: new Date()
            });

            const session = await mongoose.startSession();
            try {
                await session.withTransaction(async () => {
                    // Update queue status if exists
                    const queueStatus = await this.queueManager.getAgentQueue(agent._id);
                    if (queueStatus) {
                        queueStatus.processing_details.failed_files += 1;
                        await queueStatus.save({ session });
                    }

                    // Update file status to failed
                    await agent.update_file_status(file_url, 'failed', {
                        error: error.message
                    }, { session });
                });
            } finally {
                session.endSession();
            }

            throw new Error(`Failed to process file ${file_url}: ${error.message}`);
        }
    }

    /**
     * Gets all raw files for an agent
     * @param {string} agent_id - The ID of the agent
     * @returns {Promise<Array<Object>>} Array of file entries with status
     */
    async get_agent_raw_files(agent_id) {
        try {
            const agent = await AgentModel.findById(agent_id);
            if (!agent) {
                throw new Error('Agent not found');
            }
            return agent.raw_files;
        } catch (error) {
            console.error('Error getting agent raw files:', error);
            throw error;
        }
    }

    /**
     * Clean up OpenAI resources for an agent
     * @param {string} agent_id - The ID of the agent
     */
    async cleanup_agent_resources(agent_id) {
        const agent = await AgentModel.findById(agent_id);
        if (!agent) {
            throw new Error('Agent not found');
        }

        try {
            // Delete OpenAI files
            for (const file of agent.raw_files) {
                if (file.openai_file_id) {
                    try {
                        await this.openai_service.client.files.delete(file.openai_file_id);
                    } catch (error) {
                        console.error(`Failed to delete OpenAI file ${file.openai_file_id}:`, error);
                    }
                }
            }

            // Delete vector store if exists
            if (agent.openai_vector_store_id) {
                try {
                    await this.openai_service.client.beta.vectorStores.delete(agent.openai_vector_store_id);
                } catch (error) {
                    console.error(`Failed to delete vector store ${agent.openai_vector_store_id}:`, error);
                }
            }

            // Update agent record
            agent.raw_files.forEach(file => {
                file.openai_file_id = null;
                file.processing_status = 'failed';
            });
            agent.openai_vector_store_id = null;
            await agent.save();

        } catch (error) {
            console.error('Error cleaning up agent resources:', error);
            throw error;
        }
    }

    /**
     * Get processing status summary for an agent
     * @param {string} agent_id - The ID of the agent
     */
    async get_processing_status(agent_id) {
        const agent = await AgentModel.findById(agent_id);
        if (!agent) {
            throw new Error('Agent not found');
        }

        const status_counts = agent.raw_files.reduce((acc, file) => {
            acc[file.processing_status] = (acc[file.processing_status] || 0) + 1;
            return acc;
        }, {});

        const queueStatus = await this.queueManager.getAgentQueue(agent._id);

        return {
            total_files: agent.raw_files.length,
            status_counts,
            queue_status: agent.queue_status,
            queue_position: queueStatus?.queue_position,
            progress: queueStatus?.progress || 0,
            processing_attempts: agent.processing_attempts,
            last_processed_at: agent.last_processed_at,
            has_errors: agent.error_logs?.length > 0,
            recent_errors: agent.error_logs?.slice(-5) || []
        };
    }

    async upload_file(agentId, filePath) {
        try {
            // Verify file exists and is readable
            if (!fs.existsSync(filePath)) {
                throw new Error(`File not found: ${filePath}`);
            }

            const stream = fs.createReadStream(filePath);

            // Log file details before upload
            const stats = fs.statSync(filePath);
            console.log(`Uploading file: ${filePath}, size: ${stats.size} bytes`);

            const response = await this.openai.files.create({
                file: stream,
                purpose: 'assistants'
            });

            console.log(`File uploaded successfully, OpenAI File ID: ${response.id}`);

            return {
                file_id: response.id,
                filename: path.basename(filePath),
                size: stats.size
            };
        } catch (error) {
            console.error(`Failed to upload file ${filePath}:`, error);
            throw new Error(`OpenAI file upload failed: ${error.message}`);
        }
    }
}