import mongoose from "mongoose";

const AgentQueueSchema = new mongoose.Schema(
  {
    agent_id: {
      type: mongoose.Schema.Types.ObjectId,
      required: true,
      index: true,
    },
    exhibition_id: {
      type: mongoose.Schema.Types.ObjectId,
      required: true,
    },
    boughten_booth_id: {
      type: mongoose.Schema.Types.ObjectId,
      required: true,
    },
    queue_position: {
      type: Number,
      required: true,
    },
    progress: {
      type: Number,
      default: 0,
      min: 0,
      max: 100,
    },
    estimated_completion_time: {
      type: Date,
      required: true,
    },
    status: {
      type: String,
      enum: ["queued", "processing", "completed", "failed"],
      default: "queued",
    },
    processing_details: {
      start_time: {
        type: Date,
      },
      last_update_time: {
        type: Date,
        default: Date.now,
      },
      completion_time: {
        type: Date,
      },
      total_files: {
        type: Number,
        default: 0,
      },
      processed_files: {
        type: Number,
        default: 0,
      },
      failed_files: {
        type: Number,
        default: 0,
      },
      current_file: String,
    },
    error_details: {
      error_message: String,
      error_time: Date,
      error_type: String,
      error_stack: String,
    },
    system_info: {
      python_version: String,
      node_version: String,
      os_type: String,
      memory_usage: Number,
    },
  },
  {
    timestamps: true,
  }
);

// Index for efficient queue position updates
AgentQueueSchema.index({ status: 1, queue_position: 1 });

// Automatically remove completed/failed entries after 1 hour
AgentQueueSchema.index(
  { "processing_details.completion_time": 1 },
  {
    expireAfterSeconds: 3600,
    partialFilterExpression: {
      status: { $in: ["completed", "failed"] },
    },
  }
);

// Methods to update processing details
AgentQueueSchema.methods.start_processing = async function () {
  this.status = "processing";
  this.processing_details.start_time = new Date();
  this.processing_details.last_update_time = new Date();
  return await this.save();
};

AgentQueueSchema.methods.update_progress = async function (
  progress,
  current_file = null
) {
  this.progress = progress;
  this.processing_details.last_update_time = new Date();
  if (current_file) {
    this.processing_details.current_file = current_file;
  }
  return await this.save();
};

AgentQueueSchema.methods.complete_processing = async function (success = true) {
  this.status = success ? "completed" : "failed";
  this.progress = success ? 100 : this.progress;
  this.processing_details.completion_time = new Date();
  this.processing_details.last_update_time = new Date();
  return await this.save();
};

AgentQueueSchema.methods.log_error = async function (error) {
  this.error_details = {
    error_message: error.message,
    error_time: new Date(),
    error_type: error.name,
    error_stack: error.stack,
  };
  return await this.save();
};

AgentQueueSchema.methods.update_file_counts = async function (
  processed = 0,
  failed = 0
) {
  this.processing_details.processed_files += processed;
  this.processing_details.failed_files += failed;
  return await this.save();
};

// Static method to get next item in queue
AgentQueueSchema.statics.get_next_in_queue = async function () {
  return await this.findOne({ status: "queued" }).sort("queue_position").exec();
};

// Static method to get queue statistics
AgentQueueSchema.statics.get_queue_stats = async function () {
  const stats = await this.aggregate([
    {
      $group: {
        _id: "$status",
        count: { $sum: 1 },
        avg_processing_time: {
          $avg: {
            $cond: [
              { $eq: ["$status", "completed"] },
              {
                $divide: [
                  {
                    $subtract: [
                      "$processing_details.completion_time",
                      "$processing_details.start_time",
                    ],
                  },
                  1000, // Convert to seconds
                ],
              },
              null,
            ],
          },
        },
      },
    },
  ]);

  return stats.reduce((acc, stat) => {
    acc[stat._id] = {
      count: stat.count,
      avg_processing_time: stat.avg_processing_time,
    };
    return acc;
  }, {});
};

export const AgentQueueModel = mongoose.model("agent_queues", AgentQueueSchema);

export default AgentQueueModel;
