import path from 'path';
import crypto from 'crypto';
import fileConfig from '../config/fileValidation.js';

/**
 * Validates file uploads based on configured constraints
 */
class FileValidationMiddleware {
    /**
     * Validates a single file
     */
    static validateFile(file) {
        const errors = [];
        
        // Check if file type is supported
        const ext = path.extname(file.originalname).toLowerCase();
        const expectedMimeType = fileConfig.allowedTypes[ext];
        
        if (!expectedMimeType) {
            errors.push(`File type ${ext} is not supported`);
        } else if (file.mimetype !== expectedMimeType) {
            errors.push(`Invalid mime type. Expected ${expectedMimeType} but got ${file.mimetype}`);
        }

        // Check file size
        const maxSize = fileConfig.maxFileSizes[file.mimetype];
        if (maxSize && file.size > maxSize) {
            errors.push(`File size ${file.size} exceeds maximum allowed size of ${maxSize} bytes`);
        }

        return errors;
    }

    /**
     * Validates an array of files as a batch
     */
    static validateBatch(files) {
        const errors = [];
        
        // Check number of files
        if (files.length > fileConfig.batchLimits.maxFiles) {
            errors.push(`Batch contains ${files.length} files, exceeding limit of ${fileConfig.batchLimits.maxFiles}`);
        }

        // Check total batch size
        const totalSize = files.reduce((sum, file) => sum + file.size, 0);
        if (totalSize > fileConfig.batchLimits.maxTotalSize) {
            errors.push(`Total batch size ${totalSize} exceeds maximum allowed size of ${fileConfig.batchLimits.maxTotalSize} bytes`);
        }

        return errors;
    }

    /**
     * Calculates file checksum
     */
    static calculateChecksum(buffer) {
        return crypto.createHash('sha256').update(buffer).digest('hex');
    }

    /**
     * Middleware function for Express
     */
    static validate(req, res, next) {
        if (!req.files || req.files.length === 0) {
            return res.status(400).json({ error: 'No files uploaded' });
        }

        // Validate batch
        const batchErrors = FileValidationMiddleware.validateBatch(req.files);
        if (batchErrors.length > 0) {
            return res.status(400).json({ errors: batchErrors });
        }

        // Validate individual files
        const fileErrors = [];
        req.files.forEach(file => {
            const errors = FileValidationMiddleware.validateFile(file);
            if (errors.length > 0) {
                fileErrors.push({ file: file.originalname, errors });
            }
        });

        if (fileErrors.length > 0) {
            return res.status(400).json({ errors: fileErrors });
        }

        // Add checksums to files
        req.files.forEach(file => {
            file.checksum = FileValidationMiddleware.calculateChecksum(file.buffer);
        });

        next();
    }
}

export default FileValidationMiddleware;