# VR Expo AI Assistants - Development Guide

## Project Setup

### Prerequisites

- Node.js (v18+)
- MongoDB
- Redis
- Python 3.8+
- OpenAI API Key
- Docker (optional)

### Installation Steps

1. Clone the repository
2. Install dependencies:

   ```bash
   npm install
   pip install -r requirements.txt
   ```

3. Configure environment variables:

   ```
   MONGODB_URI=mongodb://localhost:27017/vr_expo
   REDIS_URL=redis://localhost:6379
   JWT_SECRET=your-secret-key
   OPENAI_API_KEY=your-openai-key
   ```

## Project Structure

```
vr-expo/
├── src/
│   ├── config/           # Configuration files
│   ├── controllers/      # Request handlers
│   ├── middleware/       # Express middleware
│   ├── models/          # Mongoose models
│   ├── routes/          # API routes
│   ├── services/        # Business logic
│   ├── utils/           # Utility functions
│   └── app.js           # Express app
├── scripts/
│   ├── data-processing/ # Python data processing scripts
│   └── queue-workers/   # Queue worker scripts
├── tests/              # Test files
└── docs/              # Documentation
```

## Core Components Implementation

### 1. JWT Authentication

```javascript
// src/middleware/auth.js
const jwt = require("jsonwebtoken");

const authMiddleware = async (req, res, next) => {
  try {
    const token = req.headers.authorization?.split(" ")[1];
    if (!token) throw new Error("No token provided");

    const decoded = jwt.verify(token, process.env.JWT_SECRET);
    req.user = decoded;
    next();
  } catch (error) {
    res.status(401).json({ error: "Authentication failed" });
  }
};
```

### 2. Data Pipeline Implementation

```javascript
// src/services/dataPipeline.js
const Queue = require("bull");
const dataQueue = new Queue("data-processing");

const processData = async (data) => {
  return await dataQueue.add({
    type: "process",
    data,
    options: {
      attempts: 3,
      backoff: {
        type: "exponential",
        delay: 2000,
      },
    },
  });
};
```

### 3. OpenAI Integration

```javascript
// src/services/openai.js
const { OpenAIApi } = require("openai");

const createAssistant = async (companyData) => {
  try {
    const assistant = await openai.createAssistant({
      instructions: `You are a virtual representative for ${companyData.name}.`,
      tools: [{ type: "retrieval" }],
      model: "gpt-4-turbo-preview",
    });

    return assistant.id;
  } catch (error) {
    console.error("OpenAI Assistant creation failed:", error);
    throw error;
  }
};
```

## API Implementation Examples

### 1. Agent Creation Endpoint

```javascript
// src/controllers/agent.js
const createAgent = async (req, res) => {
  try {
    const { companyData } = req.body;

    // Create OpenAI assistant
    const agentId = await createAssistant(companyData);

    // Store in database
    const agent = await Agent.create({
      companyId: req.company._id,
      agentId,
      status: {
        progress: 0,
        timeRemaining: null,
      },
    });

    res.json({ success: true, agent });
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};
```

### 2. Chat Implementation

```javascript
// src/controllers/chat.js
const handleMessage = async (req, res) => {
  try {
    const { message, type } = req.body;
    const { agentId } = req.params;

    // Handle different message types
    let processedMessage = message;
    if (type === "voice") {
      processedMessage = await convertSpeechToText(message);
    }

    // Get agent response
    const response = await getAgentResponse(agentId, processedMessage);

    // Convert response to speech if needed
    if (type === "voice") {
      response.audio = await convertTextToSpeech(response.text);
    }

    res.json(response);
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};
```

## Testing

### Unit Tests

```javascript
// tests/agent.test.js
describe("Agent Creation", () => {
  it("should create an agent successfully", async () => {
    const companyData = {
      name: "Test Company",
      description: "Test Description",
    };

    const agent = await createAgent(companyData);
    expect(agent).toHaveProperty("agentId");
    expect(agent.status.progress).toBe(0);
  });
});
```

### Integration Tests

```javascript
// tests/integration/api.test.js
describe("API Integration", () => {
  it("should process data pipeline", async () => {
    const response = await request(app)
      .post("/api/data/ingest")
      .send(testData)
      .set("Authorization", `Bearer ${token}`);

    expect(response.status).toBe(200);
    expect(response.body).toHaveProperty("jobId");
  });
});
```

## Deployment

### Docker Setup

```dockerfile
# Dockerfile
FROM node:18-alpine

WORKDIR /app

COPY package*.json ./
RUN npm install

COPY . .

EXPOSE 3000

CMD ["npm", "start"]
```

### Docker Compose

```yaml
# docker-compose.yml
version: "3"
services:
  app:
    build: .
    ports:
      - "3000:3000"
    environment:
      - MONGODB_URI=mongodb://mongo:27017/vr_expo
      - REDIS_URL=redis://redis:6379
    depends_on:
      - mongo
      - redis

  mongo:
    image: mongo:latest
    ports:
      - "27017:27017"

  redis:
    image: redis:alpine
    ports:
      - "6379:6379"
```

## Error Handling

```javascript
// src/utils/errorHandler.js
class AppError extends Error {
  constructor(message, statusCode) {
    super(message);
    this.statusCode = statusCode;
    this.status = `${statusCode}`.startsWith("4") ? "fail" : "error";
    this.isOperational = true;

    Error.captureStackTrace(this, this.constructor);
  }
}

const handleError = (err, req, res, next) => {
  err.statusCode = err.statusCode || 500;
  err.status = err.status || "error";

  res.status(err.statusCode).json({
    status: err.status,
    error: err,
    message: err.message,
    stack: process.env.NODE_ENV === "development" ? err.stack : undefined,
  });
};
```

## Monitoring and Logging

```javascript
// src/utils/logger.js
const winston = require("winston");

const logger = winston.createLogger({
  level: "info",
  format: winston.format.combine(
    winston.format.timestamp(),
    winston.format.json()
  ),
  transports: [
    new winston.transports.File({ filename: "error.log", level: "error" }),
    new winston.transports.File({ filename: "combined.log" }),
  ],
});

if (process.env.NODE_ENV !== "production") {
  logger.add(
    new winston.transports.Console({
      format: winston.format.simple(),
    })
  );
}
```
