# VR Expo AI Assistants - Architecture Documentation

## System Overview

The VR Expo AI Assistants system creates and manages AI-powered company representatives in a virtual reality exhibition environment. The system processes company data, creates OpenAI-based agents, and enables multi-modal interaction through chat interfaces.

## Core Components

### 1. Authentication System

- JWT-based authentication
- Role-based access control (Admin, Company, Visitor)
- Secure token management

### 2. AI Agent Management

```
┌──────────────┐     ┌──────────────┐     ┌──────────────┐
│   Company    │     │  OpenAI API  │     │    Agent     │
│    Data      │ --> │  Assistant   │ --> │  Instance    │
│              │     │  Creation    │     │              │
└──────────────┘     └──────────────┘     └──────────────┘
```

### 3. Data Pipeline System

```
┌─────────────┐    ┌──────────────┐    ┌────────────┐    ┌──────────────┐
│  Raw Data   │ -> │ Preprocessing│ -> │ OpenAI API │ -> │ Structured   │
│  Ingestion  │    │    Queue     │    │            │    │    JSON      │
└─────────────┘    └──────────────┘    └────────────┘    └──────────────┘
```

### 4. Queue Management

```
┌──────────────┐     ┌──────────────┐     ┌──────────────┐
│   Task       │     │   Queue      │     │   Worker     │
│  Producer    │ --> │   Service    │ --> │   Process    │
│              │     │  (Redis)     │     │              │
└──────────────┘     └──────────────┘     └──────────────┘
```

### 5. Chat System

```
┌──────────────┐     ┌──────────────┐     ┌──────────────┐
│    User      │     │  OpenAI API  │     │    Agent     │
│  Interface   │ <-> │  (Chat/TTS/  │ <-> │  Response    │
│              │     │     STT)     │     │              │
└──────────────┘     └──────────────┘     └──────────────┘
```

## Technical Stack

### Backend

- Node.js + Express.js
- MongoDB + Mongoose
- Redis (Queue Management)
- Python (Data Processing)
- OpenAI API

### Security

- JWT Authentication
- API Rate Limiting
- Request Validation
- CORS Configuration

### Data Processing

- Python Scripts
- Pandas for Data Manipulation
- OpenAI API Integration
- Queue Management System

## Database Schema

### Users Collection

```javascript
{
  _id: ObjectId,
  username: String,
  password: String (hashed),
  role: String (enum: ['admin', 'company', 'visitor']),
  createdAt: Date
}
```

### Companies Collection

```javascript
{
  _id: ObjectId,
  name: String,
  data: Object,
  agentId: String,
  status: {
    progress: Number,
    timeRemaining: Number
  }
}
```

### Chats Collection

```javascript
{
  _id: ObjectId,
  companyId: ObjectId,
  visitorId: ObjectId,
  messages: [{
    sender: String,
    content: String,
    timestamp: Date,
    type: String (enum: ['text', 'voice'])
  }]
}
```

### Queue Collection

```javascript
{
  _id: ObjectId,
  type: String,
  status: String,
  data: Object,
  priority: Number,
  createdAt: Date,
  processedAt: Date
}
```

## API Endpoints

### Authentication

- POST /api/auth/login
- POST /api/auth/register
- POST /api/auth/refresh-token

### Agent Management

- POST /api/agents/create
- GET /api/agents/status/:id
- PUT /api/agents/update/:id

### Data Pipeline

- POST /api/data/ingest
- GET /api/data/status
- POST /api/data/transform

### Chat

- POST /api/chat/create
- POST /api/chat/message
- GET /api/chat/history/:id

## Security Considerations

1. **Authentication & Authorization**

   - JWT token validation
   - Role-based access control
   - Token refresh mechanism

2. **Data Security**

   - Input validation
   - Data encryption
   - Secure storage of sensitive information

3. **API Security**

   - Rate limiting
   - Request validation
   - CORS policy

4. **OpenAI API**
   - Secure key management
   - Usage monitoring
   - Error handling

## Scaling Considerations

1. **Horizontal Scaling**

   - Containerization with Docker
   - Load balancing
   - Microservices architecture

2. **Queue Management**

   - Redis cluster
   - Worker pool scaling
   - Priority queue implementation

3. **Database Scaling**

   - MongoDB sharding
   - Caching layer
   - Read replicas

4. **Performance Optimization**
   - API response caching
   - Database indexing
   - Resource pooling
