import path from 'path';
import createError from 'http-errors';
import mongoose from 'mongoose';
import { AgentModel } from '../models/agent.js';
import { AgentFileService } from '../services/agent-file-service.js';
import { OpenAIService } from '../services/openai-service.js';
import { fileURLToPath } from 'url';
import { FileProcessor } from './processing/FileProcessor.js';
import { QueueManager } from './queue/QueueManager.js';

class DocumentProcessingController {
    constructor() {
        this.agentFileService = new AgentFileService(process.env.OPENAI_API_KEY);
        this.openaiService = new OpenAIService(process.env.OPENAI_API_KEY);
        this.fileProcessor = new FileProcessor(this.agentFileService);
        this.queueManager = new QueueManager();

        const __filename = fileURLToPath(import.meta.url);
        const __dirname = path.dirname(__filename);
        this.projectRoot = path.resolve(__dirname, '../../..');
    }

    // Add helper method for sanitizing errors
    _sanitizeError(error) {
        // Log the full error for debugging
        console.error('Original error:', error);

        // Return generic message for OpenAI-related errors
        if (error.message?.includes('openai') || error.message?.includes('OpenAI')) {
            return 'An error occurred while processing with AI services';
        }

        // Return generic message for file system errors
        if (error.message?.includes('ENOENT') || error.message?.includes('directory')) {
            return 'An error occurred while processing files';
        }

        // Generic message for other errors
        return 'An unexpected error occurred during processing';
    }

    /**
     * Process documents for an agent
     * @param {string} agentId - The agent ID
     */
    async process_documents(agentId) {
        try {
            // Get agent to verify existence
            const agent = await AgentModel.findById(agentId);
            if (!agent) {
                throw new Error('Agent not found');
            }

            let exhibitionId, boughtenBoothId;

            // Get boughten booth and exhibition details
            if (agent.ref !== 'boughten_booths') {
                throw new Error('Agent must be associated with a boughten booth');
            }

            try {
                const BoughtenBoothModel = mongoose.model('boughten_booths');
                const boughtenBooth = await BoughtenBoothModel.findById(agent.ref_id);
                if (!boughtenBooth) {
                    throw new Error(`Boughten booth not found with ID: ${agent.ref_id}`);
                }

                // Verify agent ID is properly linked in booth's agent_ids
                if (!boughtenBooth.agent_ids?.includes(agent._id)) {
                    // Auto-repair the relationship
                    await BoughtenBoothModel.findByIdAndUpdate(boughtenBooth._id, {
                        $addToSet: { agent_ids: agent._id }
                    });
                }

                boughtenBoothId = boughtenBooth._id;
                exhibitionId = boughtenbooth.exhibition_id
                    ;

                // If no exhibition ID and SKIP_EXHIBITION_CHECK is true, create a temporary ObjectId
                if (!exhibitionId && process.env.SKIP_EXHIBITION_CHECK === 'true') {
                    exhibitionId = new mongoose.Types.ObjectId();
                } else if (!exhibitionId) {
                    throw new Error(`Exhibition ID missing for boughten booth: ${agent.ref_id}`);
                }
            } catch (error) {
                throw new Error(`Failed to get boughten booth data: ${error.message}`);
            }

            const inputDir = path.join(
                this.projectRoot,
                'uploads',
                'agents',
                agentId,
                'raw_files'
            );

            const outputDir = path.join(
                this.projectRoot,
                'uploads',
                'agents',
                agentId,
                'processed_files'
            );

            // Verify that raw files exist
            if (!agent.raw_files || agent.raw_files.length === 0) {
                throw new Error('No raw files found for processing');
            }

            // Ensure input directory exists and contains files
            const fs = await import('fs/promises');
            try {
                const files = await fs.readdir(inputDir);
                if (files.length === 0) {
                    throw new Error('No files found in input directory');
                }
                console.log(`Found ${files.length} files in input directory`);
            } catch (error) {
                throw new Error(`Input directory error: ${error.message}`);
            }

            // Ensure output directory exists
            try {
                await fs.mkdir(outputDir, { recursive: true });
            } catch (error) {
                throw new Error(`Failed to create output directory: ${error.message}`);
            }

            // Calculate estimated completion time
            const estimatedCompletionTime = await this.fileProcessor.calculateEstimatedTime(inputDir);

            // Set agent to queued state
            agent.queue_status = 'queued';
            await agent.save();

            // Create queue entry with boughten booth and exhibition IDs
            const queueStatus = await this.queueManager.createQueueEntry(
                agentId,
                exhibitionId,
                boughtenBoothId,
                estimatedCompletionTime
            );

            // Set initial processing details
            queueStatus.processing_details.total_files = agent.raw_files.length;
            await queueStatus.save();

            // Update queue status to processing
            await queueStatus.start_processing();

            // Update agent status to match
            agent.queue_status = 'processing';
            await agent.save();

            try {
                // Process documents
                const result = await this.fileProcessor.processDocuments(
                    inputDir,
                    outputDir,
                    async (progress) => {
                        try {
                            // Update processed files count based on progress
                            queueStatus.processing_details.processed_files = Math.floor((progress / 100) * agent.raw_files.length);
                            await queueStatus.update_progress(progress);
                        } catch (progressError) {
                            console.error('Progress update error:', progressError);
                            // Don't throw here to allow processing to continue
                        }
                    }
                );

                console.log('Document processing result:', result);

                // Verify processed files exist in output directory
                const processedFiles = await fs.readdir(outputDir);
                if (processedFiles.length === 0) {
                    throw new Error('No processed files found in output directory');
                }
                console.log(`Found ${processedFiles.length} processed files ready for upload`);

                // Upload processed files to OpenAI
                const successfulUploads = await this.fileProcessor.processAndUploadFiles(agentId, outputDir);

                if (!successfulUploads || successfulUploads.length === 0) {
                    throw new Error('No files were successfully uploaded to OpenAI');
                }

                console.log(`Successfully uploaded ${successfulUploads.length} files to OpenAI`);

                // Make sure we have an assistant ID before proceeding
                if (!agent.openai_assistant_id) {
                    throw new Error('Agent does not have an OpenAI assistant ID');
                }

                try {
                    // Update existing assistant with new files
                    const { vector_store } = await this.openaiService.add_files_to_assistant({
                        assistant_id: agent.openai_assistant_id,
                        file_ids: successfulUploads.map(f => f.file_id)
                    });

                    // Update agent with new OpenAI vector store ID only
                    if (agent) {
                        agent.openai_vector_store_id = vector_store.id;
                        agent.queue_status = 'completed';
                        await agent.save();
                    }
                } catch (error) {
                    throw new Error(`Failed to update assistant with new files: ${error.message}`);
                }

                // Update queue status to completed after processing
                await queueStatus.complete_processing(true);

                // Update agent status to completed
                await AgentModel.findByIdAndUpdate(agentId, {
                    queue_status: 'completed'
                });

                // Update queue positions
                await this.queueManager.updateQueuePositions();

                // Clean up old queue entries
                await this.queueManager.cleanupOldEntries();

                return {
                    status: 'success',
                    message: 'Documents processed successfully',
                    agent_id: agentId
                };

            } catch (error) {
                // Enhanced error logging
                console.error('Processing error details:', {
                    error: error.message,
                    agentId,
                    inputDir,
                    outputDir
                });

                // Handle processing error
                queueStatus.status = 'failed';
                await queueStatus.log_error(error);

                // Update processing details for failed state
                queueStatus.processing_details.failed_files = agent.raw_files.length;
                queueStatus.processing_details.error_details = {
                    error_message: error.message,
                    error_location: 'document_processing'
                };
                await queueStatus.save();

                // Update agent status to failed
                await AgentModel.findByIdAndUpdate(agentId, {
                    queue_status: 'failed'
                });

                await this.queueManager.updateQueuePositions();
                throw error;
            }

        } catch (error) {
            const sanitizedMessage = this._sanitizeError(error);
            throw createError(500, sanitizedMessage);
        }
    }

    /**
     * Get the processing status for a specific agent
     */
    async get_agent_queue(agentId) {
        try {
            const status = await this.queueManager.getAgentQueue(agentId);

            if (!status) {
                return {
                    status: 'not_found',
                    message: 'No active queue entry found for this agent'
                };
            }

            const agentStatus = await this.agentFileService.get_processing_status(agentId);

            return {
                ...agentStatus,
                queue_status: status.status,
                queue_position: status.queue_position,
                progress: status.progress || 0,
                estimated_completion_time: status.estimated_completion_time,
                processing_details: status.processing_details,
                error: status.error_details?.error_message,
                error_details: status.error_details,
                exhibition_id
                    : status.exhibition_id
                ,
                boughten_booth_id: status.boughten_booth_id,
                system_info: status.system_info
            };
        } catch (error) {
            const sanitizedMessage = this._sanitizeError(error);
            throw createError(500, sanitizedMessage);
        }
    }
}

export default new DocumentProcessingController();