import { ConversationModel } from "../../models/conversation.js";
import { openai } from "../config/pipeline-config.js";
import { waitForCompletion, calculateTokens } from "../utils/chatHelpers.js";
import {
  transcribeAudio,
  synthesizeSpeech,
} from "../services/speechService.js";
import { AgentModel } from "../models/agent.js";
import { validationResult } from "express-validator";

export class ChatController {
  // Create a new conversation
  static async createConversation(req, res) {
    // Validate request fields using express-validator middleware
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return res.status(400).json({ success: false, errors: errors.array() });
    }
    try {
      const { agent_id, user_id, initial_message } = req.body;
      const agent = await AgentModel.findById(agent_id);
      if (!agent) {
        return res
          .status(404)
          .json({ success: false, error: "Agent not found" });
      }
      if (!agent.openai_assistant_id) {
        return res
          .status(400)
          .json({ success: false, error: "Agent not configured" });
      }
      const thread = await openai.beta.threads.create({
        messages: [{ role: "user", content: initial_message }],
      });
      const conversation = await ConversationModel.create({
        agent_id,
        user_id,
        thread_id: thread.id,
        messages: [{ role: "user", content: initial_message }],
        token_count: calculateTokens(initial_message),
      });
      // Respond with only the conversation id
      return res.status(201).json({
        success: true,
        data: { conversation_id: conversation._id },
      });
    } catch (error) {
      console.error("Error creating conversation:", error);
      let errorMessage = "Internal server error";
      if (
        error.message &&
        error.message.toLowerCase().includes("openai", "oepn ai")
      ) {
        errorMessage = "An error occurred with the Agent services.";
      }
      return res.status(500).json({ success: false, error: errorMessage });
    }
  }

  // Send message and get response
  static async sendMessage(req, res) {
    // Validate request fields using express-validator middleware
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return res.status(400).json({ success: false, errors: errors.array() });
    }
    try {
      const { conversation_id } = req.body;
      const message = req.message;
      console.log(message);
      const conversation = await ConversationModel.findById(conversation_id);
      if (!conversation) {
        return res
          .status(404)
          .json({ success: false, error: "Conversation not found" });
      }
      const agent = await AgentModel.findById(conversation.agent_id);
      if (!agent) {
        return res
          .status(404)
          .json({ success: false, error: "Agent not found" });
      }
      if (!agent.openai_assistant_id) {
        return res
          .status(400)
          .json({ success: false, error: "Agent not configured" });
      }
      await openai.beta.threads.messages.create(conversation.thread_id, {
        role: "user",
        content: message,
      });

      const run = await openai.beta.threads.runs.create(
        conversation.thread_id,
        {
          assistant_id: agent.openai_assistant_id,
        }
      );
      await waitForCompletion(openai, conversation.thread_id, run.id);
      const messages = await openai.beta.threads.messages.list(
        conversation.thread_id
      );
      const assistantMessage = messages.data[0].content[0].text.value;
      const mp3 = await openai.audio.speech.create({
        model: "tts-1",
        voice: "alloy",
        input: assistantMessage,
      });
      const audioBuffer = Buffer.from(await mp3.arrayBuffer());
      conversation.messages.push(
        { role: "user", content: message },
        { role: "assistant", content: assistantMessage }
      );
      conversation.token_count += calculateTokens(message + assistantMessage);
      await conversation.save();
      return res.status(200).json({
        success: true,
        data: {
          message: assistantMessage,
          conversation_id: conversation._id,
          audio: audioBuffer.toString("base64"),
        },
      });
    } catch (error) {
      console.error("Error sending message:", error);
      let errorMessage = "Internal server error";
      if (
        error.message &&
        error.message.toLowerCase().includes("openai", "oepn ai")
      ) {
        errorMessage = "An error occurred with the Agent services.";
      }
      return res.status(500).json({ success: false, error: errorMessage });
    }
  }

  // Get conversation history
  static async getConversation(req, res) {
    // Validate request fields using express-validator middleware
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return res.status(400).json({ success: false, errors: errors.array() });
    }
    try {
      const { conversation_id } = req.params;
      const conversation = await ConversationModel.findById(conversation_id);
      if (!conversation) {
        return res
          .status(404)
          .json({ success: false, error: "Conversation not found" });
      }
      return res.status(200).json({ success: true, data: conversation });
    } catch (error) {
      console.error("Error getting conversation:", error);
      let errorMessage = "Internal server error";
      if (
        error.message &&
        error.message.toLowerCase().includes("openai", "oepn ai")
      ) {
        errorMessage = "An error occurred with the Agent services.";
      }
      return res.status(500).json({ success: false, error: errorMessage });
    }
  }

  // Speech to text using Whisper
  static async transcribeAudio(req, res) {
    return transcribeAudio(req, res);
  }

  // Text to speech
  static async synthesizeSpeech(req, res) {
    return synthesizeSpeech(req, res);
  }

  // Handle text message
  static async handleTextMessage(req, res) {
    req.message = req.body.text;
    return await ChatController.sendMessage(req, res);
  }

  // Handle audio message
  static async handleAudioMessage(req, res) {
    try {
      // Transcribe audio
      const transcriptionResult = await transcribeAudio(req, res);
      const messageText = transcriptionResult;

      // Process message using existing sendMessage logic
      req.message = messageText;
      return await ChatController.sendMessage(req, res);
    } catch (error) {
      return res.status(500).json({
        success: false,
        error: error.message,
      });
    }
  }

  // Unified message handler
  static async handleUnifiedMessage(req, res) {
    try {
      const { text } = req.body;

      // Input validation - ensure exactly one input type
      if ((!text && !req.file) || (text && req.file)) {
        return res.status(400).json({
          success: false,
          error: "Exactly one input type (text or audio) must be provided",
        });
      }

      if (text) {
        return await ChatController.handleTextMessage(req, res);
      } else if (req.file) {
        return await ChatController.handleAudioMessage(req, res);
      }
    } catch (error) {
      return res.status(500).json({
        success: false,
        error: error.message,
      });
    }
  }
}
