import path from 'path';
import OpenAI from 'openai';

// Initialize OpenAI client
const openaiApiKey = process.env.OPENAI_API_KEY;
if (!openaiApiKey) {
    throw new Error('OPENAI_API_KEY environment variable is required');
}

export const openai = new OpenAI({
    apiKey: openaiApiKey
});

/**
 * Configuration validator for AI Pipeline
 */
export class PipelineConfig {
    /**
     * Validate and normalize pipeline configuration
     * @param {Object} config Raw configuration object
     * @returns {Object} Validated and normalized configuration
     * @throws {Error} If configuration is invalid
     */
    static validate(config) {
        if (!config) {
            throw new Error('Configuration object is required');
        }

        // Validate OpenAI API key
        if (!config.openaiApiKey) {
            throw new Error('OpenAI API key is required in configuration');
        }

        // Normalize upload directory
        const uploadDir = config.uploadDir || path.join(process.cwd(), 'uploads');

        return {
            openaiApiKey: config.openaiApiKey,
            uploadDir,
            environment: process.env.NODE_ENV || 'development'
        };
    }

    /**
     * Validate assistant configuration
     * @param {Object} config Assistant configuration object
     * @returns {Object} Validated assistant configuration
     * @throws {Error} If configuration is invalid
     */
    static validateAssistantConfig(config) {
        if (!config.name || typeof config.name !== 'string') {
            throw new Error('Assistant name is required and must be a string');
        }

        if (!config.instructions || typeof config.instructions !== 'string') {
            throw new Error('Assistant instructions are required and must be a string');
        }

        if (!config.model || typeof config.model !== 'string') {
            throw new Error('Assistant model is required and must be a string');
        }

        return {
            name: config.name,
            instructions: config.instructions,
            model: config.model,
            tools: Array.isArray(config.tools) ? config.tools : [],
            vectorStoreName: config.vectorStoreName || `${config.name}_store`,
            fileIds: Array.isArray(config.fileIds) ? config.fileIds : []
        };
    }
}