import { imageSize } from 'image-size';
import fs from 'fs';

const fileValidation = {
    // حداکثر حجم مجاز فایل‌ها بر حسب بایت
    maxFileSizes: {
        'application/pdf': 20 * 1024 * 1024,
        'application/msword': 15 * 1024 * 1024,
        'application/vnd.openxmlformats-officedocument.wordprocessingml.document': 15 * 1024 * 1024,
        'text/csv': 25 * 1024 * 1024,
        'application/vnd.ms-excel': 20 * 1024 * 1024,
        'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet': 20 * 1024 * 1024,
        'text/plain': 10 * 1024 * 1024,
        'text/markdown': 10 * 1024 * 1024,
        'application/yaml': 5 * 1024 * 1024,
        'application/json': 10 * 1024 * 1024,
        'image/jpeg': 5 * 1024 * 1024,
        'image/png': 5 * 1024 * 1024,
        'video/mp4': 100 * 1024 * 1024,
        'video/webm': 80 * 1024 * 1024,
        'video/quicktime': 100 * 1024 * 1024
    },

    // فرمت‌های مجاز و MIME Type متناظر آن‌ها
    allowedTypes: {
        '.pdf': 'application/pdf',
        '.doc': 'application/msword',
        '.docx': 'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
        '.csv': 'text/csv',
        '.xls': 'application/vnd.ms-excel',
        '.xlsx': 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
        '.txt': 'text/plain',
        '.md': 'text/markdown',
        '.yaml': 'application/yaml',
        '.yml': 'application/yaml',
        '.json': 'application/json',
        '.jpg': 'image/jpeg',
        '.jpeg': 'image/jpeg',
        '.png': 'image/png',
        '.mp4': 'video/mp4',
        '.webm': 'video/webm',
        '.mov': 'video/quicktime'
    },

    // محدودیت‌های کلی آپلود دسته‌ای
    batchLimits: {
        maxTotalSize: 100 * 1024 * 1024,
        maxFiles: 50,
        concurrentProcessing: 10
    },

    encoding: {
        default: 'UTF-8',
        supported: ['UTF-8', 'UTF-16', 'ASCII', 'Windows-1256'],
        fallback: 'UTF-8'
    },

    contentValidation: {
        maxLineLength: 10000,
        maxJsonNesting: 10,
        maxYamlNesting: 10,
        maxKeyLength: 100
    },

    // پشتیبانی از نویسه‌های فارسی
    persianSupport: {
        enabled: true,
        unicodeRange: {
            start: 0x0600,
            end: 0x06FF
        },
        normalizeChars: {
            'ي': 'ی',
            'ك': 'ک',
            '١': '۱',
            '٢': '۲',
            '٣': '۳',
            '٤': '۴',
            '٥': '۵',
            '٦': '۶',
            '٧': '۷',
            '٨': '۸',
            '٩': '۹',
            '٠': '۰'
        }
    },

    // ابعاد مجاز تصاویر بر اساس نوع reference
    allowedDimensions: {
        logo: { width: 1080, height: 1080 },
        posters: { width: 720, height: 1440 },
        video_walls: { width: 1440, height: 720 },
        monitors: { width: 1440, height: 720 },
        stands: { width: 1440, height: 720 }
    }
};

/**
 * بررسی متادیتا و ابعاد فایل (در صورت تصویری بودن)
 * @param {Express.Multer.File} file - فایل دریافتی
 * @param {string} ref - نوع reference مثل "logo" یا "poster"
 * @returns {{valid: boolean, error?: string}}
 */
function validateFileMetadata(file, ref) {
    const ext = file.originalname.substring(file.originalname.lastIndexOf('.')).toLowerCase();
    const allowedMime = fileValidation.allowedTypes[ext];

    if (!allowedMime) {
        return { valid: false, error: 'File type not allowed' };
    }

    if (file.mimetype !== allowedMime) {
        return { valid: false, error: 'MIME type does not match file extension' };
    }

    const maxSize = fileValidation.maxFileSizes[file.mimetype];
    if (maxSize === undefined || file.size > maxSize) {
        return { valid: false, error: 'File size exceeds limit' };
    }

    // بررسی ابعاد فایل تصویری (در صورت نیاز)
    if (file.mimetype.startsWith('image/') && fileValidation.allowedDimensions[ref]) {
        try {
            const buffer = fs.readFileSync(file.path);
            const dimensions = imageSize(buffer);
            const expected = fileValidation.allowedDimensions[ref];

            if (
                dimensions.width !== expected.width ||
                dimensions.height !== expected.height
            ) {
                return {
                    valid: false,
                    error: `Invalid image dimensions: expected ${expected.width}x${expected.height}px, got ${dimensions.width}x${dimensions.height}px`
                };
            }
        } catch (err) {
            return { valid: false, error: 'Unable to read image dimensions' };
        }
    }

    return { valid: true };
}

export { validateFileMetadata };
export default fileValidation;
